<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Watson\Validating\ValidatingTrait;

class Doctor extends Model
{
    use ValidatingTrait;

    protected $fillable = [
        'user_id',
        'merchant_id',
        'professional_id',
        'prefix',
        'contact_number',
        'bio',
        'minimum_consultation_fee',
        'maximum_consultation_fee',
    ];
    
    protected $rules = [
        'user_id'      				=> 'required|integer',
        'merchant_id'				=> 'required|integer',
        'professional_id'   		=> 'required|string',
        'prefix'            		=> 'required|string',       
        'contact_number'            => ['required','regex:/^(09|\+639)\d{9}$/'],
        'minimum_consultation_fee' 	=> 'required|integer',
        'maximum_consultation_fee'  => 'required|integer',         
    ];
    
    // Custom validation rules
    protected $validationMessages = [
        'contact_number.regex' => "Mobile number is invalid."
    ];

    public static function rules()
    {
        return (new static)->rules;
    }

    public function setContactNumberAttribute($contactNumber)
    {   
        if (\Str::startsWith($contactNumber, '9')) {
            $contactNumber = '0'.$contactNumber;
        }
        $this->attributes['contact_number'] = $contactNumber;
    }

    public function image()
    {
        return $this->belongsTo(DoctorImage::class, 'id', 'doctor_id');
    }

    public function doctorSpecialities()
    {
        return $this->hasMany(DoctorSpeciality::class, 'doctor_id');
    }

    public function doctorPaymentMethods()
    {
        return $this->hasMany(DoctorPaymentMethod::class, 'doctor_id');
    }

    public function user() {
        return $this->hasOne(User::class, 'id', 'user_id');
    }
}
