<?php

namespace App\Http\Controllers\Dashboard\Settings\MerchantEstablishmentSchedule;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\EstablishmentSchedule;
use App\Models\Establishment;

class ViewController extends Controller
{
    public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Establishment Schedules View',
            'reference_id' => null,
            'table' => 'establishment_schedules',
            'remarks' => 'Establishment Schedules View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.settings.merchant-schedules.index');
    }

    public function merchantScheduleList()
    {
    	$establishments = Establishment::select([
                        'establishments.id',
                        'establishments.name',
                        'establishments.status',
                        'establishments.created_at',
                        'establishment_images.path',
                        'establishment_schedules.establishment_id',
                        'merchant_types.name as merchant_type',
                    ])
                    ->leftJoin('establishment_schedules', 'establishment_schedules.establishment_id', 'establishments.id')
                    ->leftJoin('establishment_images', 'establishment_images.establishment_id', 'establishments.id')
                    ->leftJoin('merchant_types', 'merchant_types.id', 'establishments.merchant_type_id')
                    ->where('establishments.merchant_id', auth()->user()->account->merchant_id)
                    ->latest()
                    ->get();

        return Datatables::of($establishments)
            ->addIndexColumn()
            ->addColumn('name', function($establishment){
                $name = ucwords($establishment->name);
                $photo = (isset($establishment->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$establishment->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;
            })
            ->addColumn('merchant_type', function($establishment){
                return ucwords($establishment->merchant_type);
            })
            ->addColumn('status', function($establishment){
               	$status = $establishment->status == 1 ? 
                    '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active' :
                    '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Inactive' ;
                return $status;
            })
            ->addColumn('action', function ($establishment){
            	if ($establishment->establishment_id != null) {
            		$button = '<a href='.route('merchants.schedules.view',base64_encode($establishment->id)).' id="view">
                                <button class="btn btn-sm btn-info mr-1"> 
                                    <i class="mdi mdi-eye"></i> View
                                </button>
                            </a>';
                	$button .= '<a href='.route('merchants.schedules.edit',base64_encode($establishment->id)).' id="edit">
                                <button class="btn btn-sm btn-primary mr-1"> 
                                    <i class="mdi mdi-table-edit"></i> Edit
                                </button>
                            </a>';
            	} else {
            		$button = '<a href='.route('merchants.schedules.add',base64_encode($establishment->id)).'>
                                <button class="btn btn-sm btn-success mr-1"> 
                                    <i class="mdi mdi-plus"></i> Add Schedule
                                </button>
                            </a>';
            	}

            	$action = '<div style="display:flex">
                                '.$button.'
            			   </div>';
            	return $action;
            })
        	->escapeColumns([])->make(true);
    }

    public function add($id)
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Establishment Schedule Add',
            'reference_id' => base64_decode($id),
            'table' => 'establishment_schedules',
            'remarks' => 'Success viewing edit establishment add.',
        ];
        
        $this->addLogs($params);

    	return view('dashboard.settings.merchant-schedules.add', [
    		'establishment' => $this->getEstablishment($id)
    	]);
    }

    public function edit($id)
    {   
        $establishment = $this->getEstablishment($id);
    	$establishment['schedules'] = $this->getSchedules($establishment->id);

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Establishment Schedule Edit',
            'reference_id' => base64_decode($id),
            'table' => 'establishment_schedules',
            'remarks' => 'Success viewing edit establishment schedule.',
        ];
        
        $this->addLogs($params);

    	return view('dashboard.settings.merchant-schedules.edit', [
    		'establishment' => $establishment
    	]);
    }

    // get schedules
    public function getSchedules($id)
    {
    	$schedule = EstablishmentSchedule::where('establishment_id', $id)->first();

    	$schedules = (Object)[
    		'monday'	=> $this->formatScheduleTime($schedule->monday),
    		'tuesday' 	=> $this->formatScheduleTime($schedule->tuesday),
    		'wednesday' => $this->formatScheduleTime($schedule->wednesday),
    		'thursday' 	=> $this->formatScheduleTime($schedule->thursday),
    		'friday' 	=> $this->formatScheduleTime($schedule->friday),
    		'saturday' 	=> $this->formatScheduleTime($schedule->saturday),
    		'sunday' 	=> $this->formatScheduleTime($schedule->sunday),
    	];

    	return $schedules;
    }

    // format schedule time
    public function formatScheduleTime($time)
    {
    	$amFrom = substr($time, 1, 8);
    	$amTo = substr($time, 12, 8);
    	$pmFrom = substr($time, 25, 8);
    	$pmTo = substr($time, 36, 8);

    	$amFrom = $amFrom ? date('H:i', strtotime($amFrom)) : '';
        $amTo = $amTo ? date('H:i', strtotime($amTo)) : '';
        $pmFrom = $pmFrom ? date('H:i', strtotime($pmFrom)) : '';
        $pmTo = $pmTo ? date('H:i', strtotime($pmTo)) : '';
    	
    	return [$amFrom, $amTo, $pmFrom, $pmTo];
    }

    public function view($id)
    {	
    	$establishment = Establishment::select([
                        'establishments.id',
                        'establishments.name',
                        'establishments.address',
                        'establishments.status',
                        'establishment_images.path',
                        'establishment_schedules.monday',
                        'establishment_schedules.tuesday',
                        'establishment_schedules.wednesday',
                        'establishment_schedules.thursday',
                        'establishment_schedules.friday',
                        'establishment_schedules.saturday',
                        'establishment_schedules.sunday',
                        'merchant_types.name as merchant_type',
                    ])
                    ->leftJoin('establishment_images', 'establishment_images.establishment_id', 'establishments.id')
                    ->leftJoin('establishment_schedules', 'establishment_schedules.establishment_id', 'establishments.id')
                    ->leftJoin('merchant_types', 'merchant_types.id', 'establishments.merchant_type_id')
                    ->where('establishments.id', base64_decode($id))
                    ->first();

        $establishment['path'] = isset($establishment->path) ? config('app.AWS_BUCKET_URL').$establishment->path : 'admin_dashboard/img/default.png';

    	return $establishment;
    }

    // get establishment
    public function getEstablishment($id)
    {
        $establishment = Establishment::select([
                        'establishments.id',
                        'establishments.name',
                        'establishments.address',
                        'establishments.description',
                        'establishment_images.path',
                        'merchant_types.name as merchant_type',
                    ])
                    ->leftJoin('establishment_schedules', 'establishment_schedules.establishment_id', 'establishments.id')
                    ->leftJoin('establishment_images', 'establishment_images.establishment_id', 'establishments.id')
                    ->leftJoin('merchant_types', 'merchant_types.id', 'establishments.merchant_type_id')
                    ->where('establishments.id', base64_decode($id))
                    ->first();

        $establishment['path'] = isset($establishment->path) ? config('app.AWS_BUCKET_URL').$establishment->path : 'admin_dashboard/img/default.png'; 

        return $establishment;
    }
}
