<?php

namespace App\Http\Controllers\Dashboard\Settings\MerchantEstablishment;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\Establishment;
use App\Models\MerchantType;
use App\Models\Merchant;

class ViewController extends Controller
{
    public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Establishment View',
            'reference_id' => null,
            'table' => 'establishments',
            'remarks' => 'Success viewing all establishments.',
        ];

        $this->addLogs($params);

    	return view('dashboard.settings.merchant-establishments.index',[
            'merchant_types' => $this->getActiveMerchantTypes(),
        ]);
    }

    public function establishmentList()
    {
    	$establishments = Establishment::select([
                        'establishments.id',
                        'establishments.name',
                        'establishments.address',
                        'establishments.status',
                        'establishments.created_at',
                        'merchant_types.name as merchant_type',
                    ])
                    ->leftJoin('merchant_types', 'merchant_types.id', 'establishments.merchant_type_id')
                    ->latest()
                    ->get();

        return Datatables::of($establishments)
            ->addIndexColumn()
            ->addColumn('name', function($establishment){
                $name = ucwords($establishment->name);
                $photo = (isset($establishment->image->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$establishment->image->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;
            })
            ->addColumn('address', function($establishment){
                return ucwords($establishment->address);
            })
            ->addColumn('merchant_type', function($establishment){
                return ucwords($establishment->merchant_type);
            })
            ->addColumn('status', function($establishment){
               	$status = $establishment->status == 1 ? 
                    '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active' :
                    '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Inactive' ;
                return $status;
            })
            ->addColumn('action', function ($establishment){
                $status = $establishment->status == 1 ? 'Deactivate' : 'Activate';
                $button = '<a href='.route('settings.merchant.establishments.edit',base64_encode($establishment->id)).' id="edit">
                                <button class="btn btn-sm btn-primary mr-1"> 
                                    <i class="mdi mdi-table-edit"></i> Edit
                                </button>
                            </a>';
                $button .= $establishment->status == 1 ?
                    '<a href='.route('settings.merchant.establishments.status',base64_encode($establishment->id)).' id="status" data-status='.$status.'>
                        <button class="btn btn-sm btn-danger"> 
                            <i class="mdi mdi-minus-circle"></i> Deactivate
                        </button>
                    </a>':
                    '<a href='.route('settings.merchant.establishments.status',base64_encode($establishment->id)).' id="status" data-status='.$status.'>
                        <button class="btn btn-sm btn-success pl-3 pr-3"> 
                            <i class="mdi mdi-check-circle"></i> Activate
                        </button>
                    </a>';

            	$action = '<div style="display:flex">
                                '.$button.'
            			   </div>';
            	return $action;
            })
        	->escapeColumns([])->make(true);
    }

    public function edit($id)
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Establishment Edit',
            'reference_id' => base64_decode($id),
            'table' => 'establishments',
            'remarks' => 'Success viewing edit establishment.',
        ];
        
        $this->addLogs($params);

        $establishment = Establishment::select([
                        'establishments.*',
                        'merchants.name as merchant_name',
                    ])
                    ->leftJoin('merchants', 'establishments.merchant_id', 'merchants.id')
                    ->where('establishments.id', base64_decode($id))
                    ->first();

        $establishment['merchant_types'] = $this->getActiveMerchantTypes();

        return $establishment;
    }

    // get merchants
    public function getMerchants(Request $request)
    {
        $merchants = Merchant::select([
                    'id',
                    'merchant_type_id',
                    'name',
                ])
                ->where('merchant_type_id', $request->type)
                ->where('name', 'LIKE', '%'.$request->filter.'%')
                ->limit(10)
                ->get();

        return $merchants;
    }

    // get active merchant types
    public function getActiveMerchantTypes()
    {
        return MerchantType::whereStatus(1)->get(['id','name']);
    }
}
