<?php

namespace App\Http\Controllers\Dashboard\Report\MerchantTransaction;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

// Add Models
use App\Models\PointImage;
use App\Models\Merchant;
use App\Models\Point;
use App\Models\User;

class ViewController extends Controller
{
    public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchants Transaction Report View',
            'reference_id' => null,
            'table' => 'points, members, users and merchants',
            'remarks' => 'Merchants Transaction Report Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.reports.merchants-transaction.index');
    }

    public function merchantTransactionList(Request $request)
    {
        $points = $this->getMerchantPointCustomDateRange($request);
    	
    	return Datatables::of($points)
            ->addIndexColumn()
            ->addColumn('member_name', function($point){
                $name = $point->member_id;

                return $name;
            })
            ->addColumn('merchant_name', function($point){
                $name = ucwords($point->merchant_name);

                $photo = (isset($point->merchant->images[0]->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$point->merchant->images[0]->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;    
            })
      		->addColumn('reference', function($point){
                return $point->reference;
            })
            ->addColumn('cashier', function($point){
               
                $cash = User::select([
                        'users.id',
                        'users.firstname',
                        'users.middlename',
                        'users.lastname',
                        'users.email',
                        'users.status',
                        'merchant_cashiers.cashier_id',
                        'merchant_cashiers.employee_number',
                        'merchant_cashiers.department',
                        'merchant_cashiers.created_at',
                    ])
                    ->join('merchant_cashiers','users.id','merchant_cashiers.user_id')
                    ->where('users.usertype','merchant_cashier')
                    ->where('merchant_cashiers.user_id', $point->cashier_id)
                    ->latest()
                    ->get();


                return $cash[0]->cashier_id;           

            })
            ->addColumn('amount', function($point){
            	$amount = number_format(floor($point->amount*100)/100, 2);
                return '₱ '.$amount;
            })
             ->addColumn('rebate', function($point){
                $reb = number_format(floor($point->rebate*100)/100, 2);
                return '₱ '.$reb;
            })
            ->addColumn('points', function($point){
            	return number_format(floor($point->points*100)/100, 2);
            })
            ->addColumn('created_at', function($point){
                return date('M d,Y', strtotime($point->created_at));
            })
            ->addColumn('action', function ($point){
                $button = '<a href='.route('reports.merchants.transactions.view',base64_encode($point->id)).' id="view">
                                <button class="btn btn-sm btn-info mr-1"> 
                                    <i class="mdi mdi-eye"></i> View
                                </button>
                            </a>';
                return $button;
            })
            ->escapeColumns([])->make(true);
    }

    // get custom merchant date range
    public function getMerchantPointCustomDateRange($request)
    {   
        $points = [];
        if (isset($request->merchant_id)) {

            $date = explode("-",$request->date);
            $startDate = date('Y-m-d', strtotime($date[0]))." 00:00:00";
            $endDate = date('Y-m-d', strtotime($date[1]))." 23:59:59";

            if ($request->merchant_id != 'all') {
                $points = Point::select([
                        'users.id as user_id',
                        'users.firstname',
                        'users.middlename',
                        'users.lastname',
                        'merchants.id as merchant_id',
                        'merchants.name as merchant_name',
                        'points.id',
                        'points.reference',
                        'points.amount',
                        'points.points',
                        'points.rebate',
                        'points.created_at',
                        'points.cashier_id',
                        'members.member_id',
                    ])
                    ->leftJoin('users','points.user_id','users.id')
                    ->leftJoin('members','points.user_id','members.user_id')
                    ->leftJoin('merchants', 'points.merchant_id','merchants.id')
                    ->where('merchant_id', (int)$request->merchant_id)
                    ->whereBetween('points.created_at', [$startDate, $endDate])
                    ->get();
            } 
        } else {
            if (auth()->user()->isMerchantAdmin()) {
                $points = Point::select([
                        'users.id as user_id',
                        'users.firstname',
                        'users.middlename',
                        'users.lastname',
                        'merchants.id as merchant_id',
                        'merchants.name as merchant_name',
                        'points.id',
                        'points.reference',
                        'points.amount',
                        'points.rebate',
                        'points.points',
                        'points.created_at',
                        'points.cashier_id',
                        'members.member_id',
                    ])
                    ->leftJoin('users','points.user_id','users.id')
                    ->leftJoin('members','points.user_id','members.user_id')
                    ->leftJoin('merchants', 'points.merchant_id','merchants.id')
                    ->where('merchant_id', auth()->user()->account->merchant_id)
                    ->latest()
                    ->get();

            }
        }
        return $points;
    }

    public function view($id)
    {
        $pointImages = PointImage::where('point_id', base64_decode($id))->get();
        
        $images = [];
        foreach ($pointImages as $key => $value) {
            $images[] = [
                'id' => $key,
                'path' => config('app.AWS_BUCKET_URL').$value->path
            ];
        }

        return $images;
    }
}
