<?php

namespace App\Http\Controllers\Dashboard\Report\MerchantRedemption;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

// /Add External Classes
use Yajra\Datatables\Datatables;

// Add Models
use App\Models\PromotionLog;
use App\Models\Merchant;
use App\Models\User;

class ViewController extends Controller
{
   	public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchants Redemption Report View',
            'reference_id' => null,
            'table' => 'promotion_logs, promotions, members, users and merchants',
            'remarks' => 'Merchants Redemption Report Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.reports.merchants-redemption.index');
    }

    public function merchantRedemptionList(Request $request)
    {
    	$promotions = $this->getPromotionCustomDateRange($request);

    	return Datatables::of($promotions)
            ->addIndexColumn()
            ->addColumn('merchant_name', function($promo){
                $name = ucwords($promo->merchant_name);

                $merchant = Merchant::find($promo->id);

                $photo = (isset($merchant->images[0]->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$merchant->images[0]->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;
            })
            ->addColumn('voucher_code', function($promo){
                return $promo->voucher_code;
            })
            ->addColumn('member_name', function($promo){
                $name = ucwords($promo->user->firstname.' '.$promo->user->middlename.' '.$promo->user->lastname);

                $photo = (isset($promo->user->images[0]->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$promo->user->images[0]->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;
            })
            ->addColumn('allocation_amount_before', function($promo){
            	$amount = number_format($promo->allocation_amount_before, 2);
                return '₱ '.$amount;
            })
            ->addColumn('allocation_amount_after', function($promo){
            	$total = $promo->allocation_amount_before - $promo->points_amount;
                return '₱ '.number_format($total, 2);
            })
            ->addColumn('points_amount', function($promo){
            	$amount = number_format($promo->points_amount, 2);
                return '₱ '.$amount;
            })
            ->addColumn('cashier', function($promo){
                $name = ucwords($promo->c_firstname.' '.$promo->c_middlename.' '.$promo->c_lastname);
                return $name;
            })
            ->addColumn('created_at', function($promo){
                return $promo->created_at->format('M d, Y');
            })
            ->escapeColumns([])->make(true);
    }

    // get custom merchant date range
    public function getPromotionCustomDateRange($request)
    {   
        $promotions = [];
        if (isset($request->merchant_id)) {

            $date = explode("-",$request->date);
            $startDate = date('Y-m-d', strtotime($date[0]));
            $endDate = date('Y-m-d', strtotime($date[1]));

            if ($request->merchant_id != 'all') {

                    $promotions = PromotionLog::select([
                        'promotion_logs.id',
                        'promotion_logs.allocation_amount_before',
                        'promotion_logs.points_amount',
                        'promotion_logs.created_at',
                        'merchants.id as merchant_id',
                        'merchants.name as merchant_name',
                        'users.firstname as c_firstname',
                        'users.middlename as c_middlename',
                        'users.lastname as c_lastname',
                        'members.user_id',
                        'promotions.promo_code as voucher_code',
                    ])
                    ->leftJoin('members','promotion_logs.member_id','members.user_id')
                    ->leftJoin('promotions','promotion_logs.promo_id','promotions.id')
                    ->leftJoin('merchants', 'promotions.merchant_id', 'merchants.id')
                    ->leftJoin('users','promotion_logs.user_id','users.id')
                    ->where('merchants.id', $request->merchant_id)
                    ->whereBetween('promotion_logs.created_at', [$startDate, $endDate])
                    ->latest()
                    ->limit(1000)
                    ->get();

            } else {

                $promotions = PromotionLog::select([
                        'promotion_logs.id',
                        'promotion_logs.allocation_amount_before',
                        'promotion_logs.points_amount',
                        'promotion_logs.created_at',
                        'merchants.id as merchant_id',
                        'merchants.name as merchant_name',
                        'users.firstname as c_firstname',
                        'users.middlename as c_middlename',
                        'users.lastname as c_lastname',
                        'members.user_id',
                        'promotions.promo_code as voucher_code',
                    ])
                    ->leftJoin('members','promotion_logs.member_id','members.user_id')
                    ->leftJoin('promotions','promotion_logs.promo_id','promotions.id')
                    ->leftJoin('merchants', 'promotions.merchant_id', 'merchants.id')
                    ->leftJoin('users','promotion_logs.user_id','users.id')
                    ->whereBetween('promotion_logs.created_at', [$startDate, $endDate])
                    ->latest()
                    ->limit(1000)
                    ->get();
            }
        } else {
            if (auth()->user()->isMerchantAdmin()) {

                $promotions = PromotionLog::select([
                        'promotion_logs.id',
                        'promotion_logs.allocation_amount_before',
                        'promotion_logs.points_amount',
                        'promotion_logs.created_at',
                        'merchants.id as merchant_id',
                        'merchants.name as merchant_name',
                        'users.firstname as c_firstname',
                        'users.middlename as c_middlename',
                        'users.lastname as c_lastname',
                        'members.user_id',
                        'promotions.promo_code as voucher_code',
                    ])
                    ->leftJoin('members','promotion_logs.member_id','members.user_id')
                    ->leftJoin('promotions','promotion_logs.promo_id','promotions.id')
                    ->leftJoin('merchants', 'promotions.merchant_id', 'merchants.id')
                    ->leftJoin('users','promotion_logs.user_id','users.id')
                    ->where('promotions.merchant_id', auth()->user()->account->merchant_id)
                    ->latest()
                    ->limit(1000)
                    ->get();
            }
        }
        return $promotions;
    }
}
