<?php

namespace App\Http\Controllers\Dashboard\Report\MerchantProductSales;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

// Add Models
use App\Models\ProductCategory;
use App\Models\MerchantAgreement;
use App\Models\Product;
use App\Models\Order;

use DB;

class ViewController extends Controller
{
    public function index()
    {
    	$params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchant Product Sales Report View',
            'reference_id' => null,
            'table' => 'member_types, users and members',
            'remarks' => 'Merchant Product Sales Report Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.reports.merchant-product-sales.index', [
    		'products' 		=> $this->getProducts(),
    		'categories' 	=> $this->getProductCategories(),
    	]);
    }

    public function productSaleList(Request $request)
    {	
    	$products = $this->getProductCustomDateRange($request);

        return Datatables::of($products)
            ->addIndexColumn()
            ->addColumn('item_code', function($product){
                return $product->item_code;
            })
            ->addColumn('name', function($product){
                $name = ucwords($product->name);
                $photo = (isset($product->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$product->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })            
            ->addColumn('category', function($product){
                return $product->category;
            })
            ->addColumn('price', function($product){
                return '₱ '.$product->price;
            })
            ->addColumn('quantity', function($product){
                return $product->quantity;
            })
            ->addColumn('total_amount', function($product){
                return '₱ '.$product->total_amount;
            })
            ->addColumn('status', function($product){
                if ($product->status == 'pending') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending' ;
                } elseif ($product->status == 'accepted' || $product->status == 'prepared'){
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-info"></i> '.ucwords($product->status);
                } elseif ($product->status == 'cancelled') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Cancelled';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Completed';
                }
                return $status;
            })
            ->escapeColumns([])->make(true);
    }

    // get custom product date range
    public function getProductCustomDateRange($request)
    {	
    	if (isset($request->status)) {
            $status_array = $request->status;
            $category_array = $request->categories;
            $product_array = $request->items;
    		if(in_array("all", $status_array)) {
                unset($status_array[0]);
            }
            if(in_array("all", $category_array)) {
                unset($category_array[0]);
            }

            if(in_array("all", $product_array)) {
                unset($product_array[0]);
            }

    		$date = explode("-",$request->date);
			$startDate = date('Y-m-d', strtotime($date[0]));
			$endDate = date('Y-m-d', strtotime($date[1]));

			$products = Order::select([
						'product_images.path',
						'products.item_code',
						'products.name',
						'products.price',
						'product_categories.name as category',
						'orders.created_at',
						'transaction_orders.status',
		                DB::raw("FORMAT(SUM(orders.quantity), 'N', 'en-us') AS quantity"), 
		                DB::raw("FORMAT(SUM(products.price * orders.quantity), 'N', 'en-us') AS total_amount"), 
		            ])
		            ->leftJoin('transaction_orders', 'transaction_orders.id', 'orders.transaction_order_id')
		            ->leftJoin('products', 'products.id', 'orders.product_id')
		            ->leftJoin('product_images', 'product_images.product_id', 'products.id')
		            ->leftJoin('product_categories', 'product_categories.id', 'products.product_category_id')
                    ->leftJoin('establishments', 'establishments.id', 'products.establishment_id')
		            ->where('establishments.merchant_id', auth()->user()->account->merchant_id)
	            	->whereBetween('orders.created_at', [$startDate, $endDate])
	                ->whereIn('transaction_orders.status', $status_array)
	                ->whereIn('products.id', $product_array)
	                ->whereIn('products.product_category_id', $category_array)
		            ->groupBy('products.id','product_images.path','products.item_code','products.name','products.price','product_categories.name','orders.created_at','transaction_orders.status')
		            ->latest()
		            ->get();
    	} else {

			$products = Order::select([
						'product_images.path',
						'products.item_code',
						'products.name',
						'products.price',
						'product_categories.name as category',
						'orders.created_at',
						'transaction_orders.status',
		                DB::raw("FORMAT(SUM(orders.quantity), 'N', 'en-us') AS quantity"), 
		                DB::raw("FORMAT(SUM(products.price * orders.quantity), 'N', 'en-us') AS total_amount"), 
		            ])
		            ->leftJoin('transaction_orders', 'transaction_orders.id', 'orders.transaction_order_id')
		            ->leftJoin('products', 'products.id', 'orders.product_id')
		            ->leftJoin('product_images', 'product_images.product_id', 'products.id')
		            ->leftJoin('product_categories', 'product_categories.id', 'products.product_category_id')
                    ->leftJoin('establishments', 'establishments.id', 'products.establishment_id')
                    ->where('establishments.merchant_id', auth()->user()->account->merchant_id)
		            ->groupBy('products.id','product_images.path','products.item_code','products.name','products.price','product_categories.name','orders.created_at','transaction_orders.status')
		            ->limit(1000)
		            ->latest()
		            ->get();
    	}
    	return $products;
    }

    // get products
    public function getProducts()
    {
        $merchant_agreement = MerchantAgreement::where('merchant_id', auth()->user()->account->merchant_id)->first();
    	return Product::select('id','name')->where('establishment_id', $merchant_agreement->establishment_id)->get();
    }

    // get product categories
    public function getProductCategories()
    {
    	return ProductCategory::select('id','name')->where('merchant_id', auth()->user()->account->merchant_id)->get();
    }
}
