<?php

namespace App\Http\Controllers\Dashboard\Report\MemberTransaction;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

// Add Models
use App\Models\TransactionOrder;
use App\Models\Member;
use App\Models\User;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            "user_id" => auth()->user()->id,
            "activity" => "Members Registration Report View",
            "reference_id" => null,
            "table" => "member_types, users and members",
            "remarks" => "Members Registration Report Success",
        ];

        $this->addLogs($params);

        return view("dashboard.reports.members-transaction.index");
    }

    public function transactionReportList(Request $request)
    {
        $members = $this->customStatusTransaction($request);

        return Datatables::of($members)
            ->addIndexColumn()
            ->addColumn("member_id", function ($member) {
                return $member->member_id;
            })
            ->addColumn("member_name", function ($member) {
                return ucwords(
                    $member->firstname .
                        " " .
                        $member->middlename .
                        " " .
                        $member->lastname
                );
            })

            ->addColumn("reference_id", function ($member) {
                return $member->reference_id;
            })
            ->addColumn("payment_method", function ($member) {
                return $member->payment_method;
            })

            ->addColumn("delivery_amount", function ($member) {
                return $member->delivery_amount;
            })
            ->addColumn("address", function ($member) {
                return $member->address;
            })
            ->addColumn("status", function ($transaction) {
                if ($transaction->status == "pending") {
                    $status =
                        '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending';
                } elseif ($transaction->status == "pending_customer") {
                    $status =
                        '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending Customer';
                } elseif (
                    $transaction->status == "accepted" ||
                    $transaction->status == "prepared"
                ) {
                    $status =
                        '<i class="mdi mdi-checkbox-blank-circle text-info"></i> ' .
                        ucwords($transaction->status);
                } elseif ($transaction->status == "cancelled") {
                    $status =
                        '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Cancelled';
                } elseif ($transaction->status == "otw") {
                    $status =
                        '<i class="mdi mdi-checkbox-blank-circle text-primary"></i> On The Way';
                } elseif ($transaction->status == "arrived") {
                    $status =
                        '<i class="mdi mdi-checkbox-blank-circle text-secondary"></i> Arrived';
                } else {
                    $status =
                        '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Completed';
                }
                return $status;
            })
            ->escapeColumns([])
            ->make(true);
    }

    public function customStatusTransaction($request)
    {
        if (isset($request->date)) {
            $date = explode("-", $request->date);
            $startDate = date("Y-m-d", strtotime($date[0]));
            $endDate = date("Y-m-d", strtotime($date[1]));

            $transactions = TransactionOrder::select([
                "users.firstname",
                "users.middlename",
                "users.lastname",
                "members.member_id",
                "user_images.path",
                "transaction_orders.id",
                "transaction_orders.establishment_id",
                "transaction_orders.reference_id",
                "transaction_orders.address",
                "transaction_orders.status",
                "transaction_orders.created_at",
                "transaction_orders.updated_at",
                "transaction_orders.payment_method",
                "transaction_orders.delivery_amount",
            ])

                ->leftJoin(
                    "members",
                    "members.user_id",
                    "transaction_orders.user_id"
                )
                ->leftJoin("users", "users.id", "transaction_orders.user_id")
                ->leftJoin("user_images", "user_images.auth_by", "users.id")
                ->whereBetween("transaction_orders.created_at", [
                    $startDate,
                    $endDate,
                ])
                // ->latest()
                ->get();
        } else {
            $transactions = TransactionOrder::select([
                "users.firstname",
                "users.middlename",
                "users.lastname",
                "members.member_id",
                "user_images.path",
                "transaction_orders.id",
                "transaction_orders.establishment_id",
                "transaction_orders.reference_id",
                "transaction_orders.address",
                "transaction_orders.status",
                "transaction_orders.created_at",
                "transaction_orders.updated_at",
                "transaction_orders.payment_method",
                "transaction_orders.delivery_amount",
            ])
                ->leftJoin(
                    "members",
                    "members.user_id",
                    "transaction_orders.user_id"
                )
                ->leftJoin("users", "users.id", "transaction_orders.user_id")
                ->leftJoin("user_images", "user_images.auth_by", "users.id")
                ->latest()
                ->get();
        }
        return $transactions;
    }

    public function formatCardNumber($card_number)
    {
        return implode("-", str_split($card_number, 4));
    }
}
