<?php

namespace App\Http\Controllers\Dashboard\Promo;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\MemberType;
use App\Models\Member;
use App\Models\Promotion;

class ViewController extends Controller
{
    public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Promos and Offers View',
            'reference_id' => null,
            'table' => 'promotions',
            'remarks' => 'Promos and Offers View Success',
        ];

        $this->addLogs($params);

        if (auth()->user()->isMerchantAdmin()) {
            return view('dashboard.merchants.promos.index', [
                'id'  => base64_encode(auth()->user()->account->merchant_id),
            ]);
        }

    	return view('dashboard.promos.index');
    }

    public function promoList()
    {   
        $promotions = Promotion::select([
                'promotions.*',
                'merchants.name',
                'merchant_images.path',
                'promotion_images.path as banner'
            ])
            ->leftJoin('merchants', 'promotions.merchant_id', 'merchants.id')
            ->leftJoin('merchant_images', 'promotions.merchant_id', 'merchant_images.merchant_id')
            ->leftJoin('promotion_images', function($join){
                $join->on('promotions.id', '=', 'promotion_images.promo_id')
                    ->where('promotion_images.type', 'banner');
            })
            ->where('promotions.status', 0)
            ->latest()
            ->get();
        
        return Datatables::of($promotions)
            ->addIndexColumn()
            ->addColumn('checkbox', function($promo){
                $checkbox = '<input class="mt-3" name="promo[]" value='.base64_encode($promo->id).' type="checkbox">';
                return $checkbox;           
            })
            ->addColumn('company', function($promo){
                $name = ucwords($promo->name);
                $photo = (isset($promo->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$promo->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src="admin_dashboard/img/default.png" class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;    
            })
            ->addColumn('promo_code', function($promo){
                $promo_code = strtoupper($promo->promo_code);
                $photo = (isset($promo->banner)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$promo->banner.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src="admin_dashboard/img/default.png" class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$promo_code;
            })
            ->addColumn('description', function($promo){
                return $promo->description;
            })
            ->addColumn('item_type', function($promo){
                return str_replace('_', ' ', strtoupper($promo->item_type));
            })
            ->addColumn('quantity', function($promo){
                return number_format($promo->quantity);
            })
            ->addColumn('status', function($promo){
                if ($promo->status == 0) {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending';
                } elseif($promo->status == 1) {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Approved';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Rejected';
                }
                return $status;
            })
            ->addColumn('created_at', function($promo){
                return $promo->created_at->format('M d,Y h:i:s A');
            })
            ->escapeColumns([])->make(true);
    }

    public function getActiveMemberTypes($id = null)
    {
        if ($id) {
            return MemberType::whereStatus(1)->orWhere('id', $id)->get(['id','name']);
        }
        return MemberType::whereStatus(1)->get(['id','name']);
    }

    public function getMembers(Request $request)
    {
        $members = Member::select([
                    'members.id',
                    'user_images.path',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                ])
                ->join('users', 'members.user_id', 'users.id')
                ->leftJoin('user_images', 'users.id', 'user_images.auth_by')
                ->where( function($query) use ($request){
                    $query->orWhere('firstname', 'LIKE', '%'.$request->filter.'%')
                        ->orWhere('middlename', 'LIKE', '%'.$request->filter.'%')
                        ->orWhere('lastname', 'LIKE', '%'.$request->filter.'%');
                })
                ->limit(10)
                ->get();
                
        return $members;
    }
}
