<?php

namespace App\Http\Controllers\Dashboard\Merchant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;


//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\MerchantAreaCode;
use App\Models\PaymentMethod;
use App\Models\Establishment;
use App\Models\MerchantType;
use App\Models\PointSetting;
use App\Models\Merchant;

class ViewController extends Controller
{
    public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchants View',
            'reference_id' => null,
            'table' => 'merchants',
            'remarks' => 'Merchants View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.index');
    }

    public function merchantList()
    {
        $merchants = Merchant::select([
                    'merchants.*',
                    'merchant_types.name as type',
                ])
                ->join('merchant_types','merchants.merchant_type_id','merchant_types.id')
                ->latest()
                ->get();

        return Datatables::of($merchants)
            ->addIndexColumn()
             ->addColumn('name', function($merchant){
                $name = ucwords($merchant->name);
                $photo = (isset($merchant->images[0]->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$merchant->images[0]->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src="admin_dashboard/img/default.png" class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;             
            })
            ->addColumn('email', function($merchant){
                return $merchant->email;
            })            ->addColumn('merchant_type', function($merchant){
                return $merchant->type;
            })
            ->addColumn('contact_number', function($merchant){
                return $merchant->contact_number;
            })
            ->addColumn('action', function ($merchant){
                $button = '<a href='.route('merchants.view',base64_encode($merchant->id)).' class="dropdown-item" id="view">
                                <i class="ti-eye"></i> View
                            </a>';
                if (auth()->user()->isSuperAdmin() || auth()->user()->isAdmin()) {
                    $button .= '<a href='.route('merchants.edit',base64_encode($merchant->id)).' class="dropdown-item" id="edit">
                                    <i class="ti-pencil"></i> Edit
                                </a>';
                    $button .= '<div class="dropdown-divider"></div>';
                    $button .= '<a href='.route('merchants.establishments',base64_encode($merchant->id)).' class="dropdown-item" id="settings">
                                    <i class="ti-layout-accordion-merged"></i> Establishments
                                </a>';
                    $button .= '<a href='.route('merchants.view.pointsetting',base64_encode($merchant->id)).' class="dropdown-item" id="points">
                                    <i class="ti-gift"></i> Loyalty Points
                                </a>';
                    $button .= '<a href='.route('merchants.settings',base64_encode($merchant->id)).' class="dropdown-item" id="settings">
                                    <i class="ti-settings"></i> Settings
                                </a>';
                }
                $action = '<div class="btn-group">
                                <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    <i class="ti-view-list-alt"></i> Options
                                </button>
                                <div class="dropdown-menu dropdown-menu-right">
                                    '.$button.'
                                </div>
                            </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function add()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchant Add',
            'reference_id' => null,
            'table' => 'merchants and users',
            'remarks' => 'Successfully viewed Add merchant page.',
        ];
        
        $this->addLogs($params);

    	return view('dashboard.merchants.add',[
            'merchant_types' => $this->getActiveMerchantTypes(),
            'area_codes'     => $this->getActiveMerchantAreaCodes(),
            'establishments' => $this->getActiveEstablishment(),
        ]);
    }

    public function rebates()
    {
        $token = config('app.calculator_token');        
        $externallogin = config('app.calculator_login_url');
        $redirecturl = config('app.calculator_redirect');

        $response = Http::withToken($token)->post($externallogin);      
   	    return redirect($redirecturl);
    }

    public function edit($id)
    {   
        $merchant = Merchant::select([
                    'merchants.*',
                    'merchant_agreements.id as merchant_agreement_id',
                    'merchant_agreements.merchant_id',
                    'merchant_agreements.establishment_id',
                    'merchant_agreements.application_type',
                    'merchant_agreements.account_type',
                    'merchant_agreements.service_fee',
                    'merchant_agreements.other_fee',
                ])
                ->leftJoin('merchant_agreements','merchants.id','merchant_agreements.merchant_id')
                ->where('merchants.id', base64_decode($id))
                ->first();

        $merchant['path'] = isset($merchant->images[0]) ? config('app.AWS_BUCKET_URL').$merchant->images[0]->path : '';
        $merchant['payment_methods'] = $this->getPaymentMethod($merchant->merchant_agreement_id);

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchant Edit',
            'reference_id' => $merchant->user_id,
            'table' => 'merchants and users',
            'remarks' => 'Successfully viewed Edit merchant page.',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.edit',[
            'merchant'       => $merchant,
            'merchant_types' => $this->getActiveMerchantTypes($merchant->merchant_type_id),
            'area_codes'     => $this->getActiveMerchantAreaCodes($merchant->merchant_area_code_id),
            'establishments' => $this->getActiveEstablishment($merchant),
        ]);
    }

    public function view($id)
    {
        $merchant = Merchant::select([
                        'merchants.*',
                        'merchant_types.name as type',
                        'merchant_area_codes.area_name',
                        'point_settings.rebate',
                        'point_settings.system',
                        'point_settings.accumulated',
                        'point_settings.advocacy',
                        'point_settings.member',
                        'point_settings.upline',
                        'point_settings.upline_number',
                        'merchant_agreements.application_type'
                    ])
                    ->leftJoin('merchant_types','merchants.merchant_type_id','merchant_types.id')
                    ->leftJoin('merchant_area_codes','merchants.merchant_area_code_id','merchant_area_codes.id')
                    ->leftJoin('point_settings', function ($join) {
                        $join->on('merchants.id', '=', 'point_settings.merchant_id')
                             ->where('point_settings.status', 1);
                    })
                    ->leftJoin('merchant_agreements', 'merchants.id', 'merchant_agreements.merchant_id')
                    ->where('merchants.id',base64_decode($id))
                    ->first();

        $merchant['date_startFormat'] = $merchant->date_start->format('M d, Y');
        $merchant['date_endFormat'] = $merchant->date_end->format('M d, Y');
        $merchant['path'] = isset($merchant->images[0]) ? config('app.AWS_BUCKET_URL').$merchant->images[0]->path : 'admin_dashboard/img/default.png'; 

        return $merchant;
    }

    // get active settings point settings
    public function getActivePointSetting($id)
    {
        $pointSetting = PointSetting::select([
                            'merchants.name',
                            'point_settings.*',
                        ])
                        ->join('merchants','merchants.id','point_settings.merchant_id')
                        ->where('point_settings.merchant_id', base64_decode($id))
                        ->whereStatus(1)
                        ->first();

        if (is_null($pointSetting)) {
            $pointSetting = ['merchant_id' => base64_decode($id)];
        }
        
        return $pointSetting; 
    }

    public function getActiveMerchantTypes($id = null)
    {
        if ($id) {
            return MerchantType::whereStatus(1)->orWhere('id', $id)->get(['id','name']);
        }
        return MerchantType::whereStatus(1)->get(['id','name']);
    }

    public function getActiveMerchantAreaCodes($id = null)
    {   
        if ($id) {
            return MerchantAreaCode::whereStatus(1)->orWhere('id', $id)->get(['id','area_name']);
        }
        return MerchantAreaCode::whereStatus(1)->get(['id','area_name']);
    }

    public function getActiveEstablishment($merchant = null)
    {
        if ($merchant) {
            $establishments = Establishment::select([
                    'establishments.id',
                    'establishments.name',
                    'establishments.merchant_id',
                    'establishment_images.path',
                    'establishments.status'
                ])
                ->join('establishment_images','establishments.id','establishment_images.establishment_id')
                ->where(['establishments.merchant_id' => $merchant->id, 'establishments.status' => 1])
                ->orWhere('establishments.id', $merchant->establishment_id)
                ->get();

        } else {
            $establishments = Establishment::select([
                    'establishments.id',
                    'establishments.name',
                    'establishments.merchant_id',
                    'establishment_images.path',
                    'establishments.status'
                ])
                ->join('establishment_images','establishments.id','establishment_images.establishment_id')
                ->where('establishments.status', 1)
                ->get();
        }
        return $establishments;
    }

    public function getPaymentMethod($id)
    {
        $paymentMethods = PaymentMethod::where('merchant_agreement_id', $id)->get();

        $data = [];
        foreach ($paymentMethods as $payment) {
            $data[] = $payment->name;
        }
        
        return $data;
    }
}
