<?php

namespace App\Http\Controllers\Dashboard\Merchant\Settings;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\PointSetting;
use App\Models\PromoMemberType;
use App\Models\PromoMember;
use App\Models\Promotion;
use App\Models\Merchant;

class ViewController extends Controller
{
    public function index($id)
    {	
        $merchant = Merchant::select([
                    'merchants.*',
                    'merchant_types.name as type',
                    'point_settings.rebate',
                    'point_settings.system',
                    'point_settings.accumulated',
                    'point_settings.advocacy',
                    'point_settings.member',
                    'point_settings.upline',
                    'point_settings.upline_number',
                    'point_settings.status',
                ])
                ->leftJoin('merchant_types','merchants.merchant_type_id','merchant_types.id')
                ->leftJoin('point_settings', function ($join) {
                    $join->on('merchants.id', '=', 'point_settings.merchant_id')
                         ->where('point_settings.status', 1);
                })
                ->where('merchants.id',base64_decode($id))
                ->first();

        $merchant['path'] = isset($merchant->images[0]) ? config('app.AWS_BUCKET_URL').$merchant->images[0]->path : 
                asset("admin_dashboard/img/default.png");

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchant Settings',
            'reference_id' => base64_decode($id),
            'table' => 'merchants and point_settings',
            'remarks' => 'Success viewing merchants settings.',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.settings.index',[
            'merchant'  => $merchant,
        ]);
    }

    public function settingList($id, $type = null)
    {   
        if ( $type != null && $type != "All") {
            $promotions = Promotion::select([
                    'promotions.*',
                    'merchants.name',
                    'promotion_images.path as banner'
                ])
                ->leftJoin('merchants', 'promotions.merchant_id', 'merchants.id')
                ->leftJoin('promotion_images', function($join){
                    $join->on('promotions.id', '=', 'promotion_images.promo_id')
                        ->where('promotion_images.type', 'banner');
                })
                ->where('promotions.merchant_id', base64_decode($id))
                ->where('status', $type)
                ->orderBy('status','desc')
                ->latest()
                ->get();
        }else{
            $promotions = Promotion::select([
                    'promotions.*',
                    'merchants.name',
                    'promotion_images.path as banner'
                ])
                ->leftJoin('merchants', 'promotions.merchant_id', 'merchants.id')
                ->leftJoin('promotion_images', function($join){
                    $join->on('promotions.id', '=', 'promotion_images.promo_id')
                        ->where('promotion_images.type', 'banner');
                })
                ->where('promotions.merchant_id', base64_decode($id))
                ->orderBy('status','desc')
                ->latest()
                ->get();
        }
       
        return Datatables::of($promotions)
            ->addIndexColumn()
            ->addColumn('promo_code', function($promo){
                $promo_code = strtoupper($promo->promo_code);
                $photo = (isset($promo->banner)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$promo->banner.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src="'.asset('admin_dashboard/img/default.png').'" class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$promo_code;
            })
            ->addColumn('item_type', function($promo){
                return str_replace("_", " ", strtoupper($promo->item_type));
            })
            ->addColumn('quantity', function($promo){
                return number_format($promo->quantity);
            })
            ->addColumn('description', function($promo){
                return $promo->description;
            })
            ->addColumn('date_start', function($promo){
                return $promo->date_start->format('M d,Y');                
            })
            ->addColumn('date_end', function($promo){
                return $promo->date_end->format('M d,Y');                
            })
            ->addColumn('status', function($promo){
                if ($promo->status == 0) {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending';
                } elseif($promo->status == 1) {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Approved';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Rejected';
                }
                return $status;
            })
            ->addColumn('action', function ($promo){
                if (auth()->user()->isMerchantAdmin() && $promo->status != 1) {
                    $button = '<a href='.route('merchants.settings.view',base64_encode($promo->id)).' class="dropdown-item" id="view">
                                    <i class="ti-eye"></i> View
                                </a>';
                    $button .= '<a href='.route('merchants.settings.edit',base64_encode($promo->id)).' class="dropdown-item" id="edit">
                                    <i class="ti-pencil"></i> Edit
                                </a>';
                    $action = '<div class="btn-group">
                                <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    <i class="ti-view-list-alt"></i> Options
                                </button>
                                <div class="dropdown-menu dropdown-menu-right">
                                    '.$button.'
                                </div>
                            </div>';
                } else {
                    $action = '<a href='.route('merchants.settings.view', base64_encode($promo->id)).' id="view">
                                <button class="btn btn-sm btn-info mr-1"> 
                                    <i class="mdi mdi-eye"></i> View
                                </button>
                            </a>';
                }
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function edit($id)
    {   
        $promo = Promotion::find(base64_decode($id));
        $promo['date_startFormat'] = $promo->date_start->format('m/d/Y');
        $promo['date_endFormat'] = $promo->date_end->format('m/d/Y');

        if ($promo->is_custom == 1) {

            if ($promo->custom_type == 'promo_members') {
                $promo['promo_members'] = $this->getPromoMembers($promo->id);
            } else {
                $promo['promo_members_types'] = $this->getPromoMemberTypes($promo->id);
            }
        }

        return $promo;
    }

    public function getPromoMembers($promo_id)
    {
        $promoMembers = PromoMember::select([
                        'promo_members.member_id as id',
                        'users.firstname',
                        'users.lastname',
                        'user_images.path',
                    ])
                    ->join('members', 'promo_members.member_id', 'members.id')
                    ->join('users', 'members.user_id', 'users.id')
                    ->leftJoin('user_images', 'members.user_id', 'user_images.auth_by')
                    ->where('promo_members.promo_id', $promo_id)
                    ->get();

        return $promoMembers;
    }

    public function getPromoMemberTypes($promo_id)
    {
        $promoMemberTypes = PromoMemberType::select([
                        'promo_member_types.member_type_id as id',
                        'member_types.name',
                    ])
                    ->join('member_types', 'promo_member_types.member_type_id', 'member_types.id')
                    ->where('promo_member_types.promo_id', $promo_id)
                    ->get();

        return $promoMemberTypes;
    }

    public function view($id)
    {
        $promo = Promotion::where('promotions.id', base64_decode($id))->first();
                
        $images = [];
        foreach ($promo->promoImages as $image) {
           array_push($images, $image->getUrlAttribute());
        }

        $promo['images'] = $images;
        $promo['item_type'] = str_replace("_", " ", strtoupper($promo->item_type));
        $promo['date_startFormat'] = $promo->date_start->format('M d,Y');
        $promo['date_endFormat'] = $promo->date_end->format('M d,Y');

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Promo View',
            'reference_id' => base64_decode($id),
            'table' => 'promotions',
            'remarks' => 'Success viewing promo.',
        ];

        $this->addLogs($params);

        return $promo;
    }

    public function getPointSettings($id)
    {
        $pointSettings = PointSetting::where('merchant_id',base64_decode($id))
                ->latest()
                ->limit(5)
                ->get();

        return Datatables::of($pointSettings)
            ->addIndexColumn()
            ->addColumn('rebate', function($point){
                return $point->rebate.'%';                
            })
            ->addColumn('system', function($point){
                return $point->system.'%';                
            })
            ->addColumn('rebate', function($point){
                return $point->rebate.'%';                
            })
            ->addColumn('accumulated', function($point){
                return $point->accumulated.'%';                
            })
            ->addColumn('advocacy', function($point){
                return $point->advocacy.'%';                
            })
            ->addColumn('member', function($point){
                return $point->member.'%';                
            })
            ->addColumn('upline', function($point){
                return $point->upline.'%';                
            })
            ->addColumn('status', function($point){
                if ($point->status == 1) {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Not Active';
                }
                return $status;
            })
            ->addColumn('created_at', function($point){
                return $point->created_at->format('M d,Y h:i:s A');                
            })
            ->escapeColumns([])->make(true);
    }
}
