<?php

namespace App\Http\Controllers\Dashboard\Merchant\Review;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\MerchantReview;
use App\Models\MerchantAgreement;
use App\Models\DoctorReview;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Review View',
            'reference_id' => null,
            'table' => 'doctors',
            'remarks' => 'Review View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.reviews.index');
    }

    public function reviewList()
    {
    	$reviews = $this->getMerchantByApplicationType();

        return Datatables::of($reviews)
            ->addIndexColumn()
            ->addColumn('customer_name', function($review){
                $name = ucwords($review->customer_firstname.' '.$review->customer_middlename.' '.$review->customer_lastname);
                $photo = (isset($review->customer_image)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$review->customer_image.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('rate', function($review){
                $active = ($review->rate >= 5) ? 5 : $review->rate;
                $inactive = 5 - (($review->rate >= 5) ? 5 : $review->rate);
                $activeStar = str_repeat('<i class="mdi mdi-star mdi-18px text-warning"></i>', $active);
                $inactiveStar = str_repeat('<i class="mdi mdi-star mdi-18px"></i>', $inactive);

                return $activeStar.$inactiveStar;
            })
            ->addColumn('comment', function($review){
                return $review->comment;
            })
            ->addColumn('doctor_name', function($review){
                $name = $review->doctor_firstname ? ucwords($review->doctor_firstname.' '.$review->doctor_middlename.' '.$review->doctor_lastname) : '';
                $photo = (isset($review->doctor_image)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$review->doctor_image.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('created_at', function($review){
                return $review->created_at->format('M d,Y h:i:s A');
            })
            ->escapeColumns([])->make(true);
    }

    public function getMerchantByApplicationType()
    {   
        if (auth()->user()->account->merchant->merchantAgreement->application_type == 'kol-a') { 
            $reviews = DoctorReview::select([
                    'doctor_reviews.rate',
                    'doctor_reviews.comment',
                    'doctor_reviews.created_at',
                    'doctor_images.path as doctor_image',
                    'users.firstname as doctor_firstname',
                    'users.middlename as doctor_middlename',
                    'users.lastname as doctor_lastname',
                    'user_images.path as customer_image',
                    'customer.firstname as customer_firstname',
                    'customer.middlename as customer_middlename',
                    'customer.lastname as customer_lastname',
                ])
                ->join('doctor_appointments', 'doctor_appointments.id', 'doctor_reviews.appointment_id')
                ->leftJoin('doctors', 'doctors.id', 'doctor_appointments.doctor_id')
                ->leftJoin('doctor_images', 'doctor_images.doctor_id', 'doctors.id')
                ->leftJoin('users', 'users.id', 'doctors.user_id')                
                ->leftJoin('users as customer', 'customer.id', 'doctor_appointments.user_id')
                ->leftJoin('user_images', 'user_images.auth_by', 'customer.id')
                ->where('doctors.merchant_id', auth()->user()->account->merchant_id)
                ->latest()
                ->get();
        } else {
            $merchant_agreement = MerchantAgreement::where('merchant_id', auth()->user()->account->merchant_id)->first();

            $reviews = MerchantReview::select([
                        'user_images.path as customer_image',
                        'users.firstname as customer_firstname',
                        'users.middlename as customer_middlename',
                        'users.lastname as customer_lastname',
                        'merchant_reviews.rate',
                        'merchant_reviews.comment',
                        'merchant_reviews.created_at',
                    ])
                    ->leftJoin('members', 'members.user_id', 'merchant_reviews.member_id')
                    ->leftJoin('users', 'users.id', 'members.user_id')
                    ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                    ->where('merchant_reviews.merchant_id', (int)$merchant_agreement->establishment_id)
                    ->latest()
                    ->get();
        }
        
        return $reviews;
    }
}
