<?php

namespace App\Http\Controllers\Dashboard\Merchant\Product;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\ProductCategory;
use App\Models\Establishment;
use App\Models\Product;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Products View',
            'reference_id' => null,
            'table' => 'products',
            'remarks' => 'Products View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.products.index',[
            'categories'        => $this->getActiveProductCategory(),
            'establishments'    => $this->getActiveMerchantEstablishment(),
        ]);
    }

    public function productList()
    { 
    	if (auth()->user()->isSuperadmin()) {
            $products = Product::select([
                    'products.id',
                    'products.name',
                    'products.description',
                    'products.price',
                    'products.status',
                    'products.created_at',
                    'product_categories.name as category_name',
                    'product_images.path',
                ])
                ->join('product_categories','product_categories.id','products.product_category_id')
                ->leftJoin('product_images', 'product_images.product_id', 'products.id')
                ->where('products.status', 0)
                ->latest()
                ->get();
        } else {
            $products = Product::select([
                    'products.id',
                    'products.establishment_id',
                    'products.name',
                    'products.description',
                    'products.price',
                    'products.status',
                    'products.created_at',
                    'product_categories.name as category_name',
                    'product_images.path',
                ])
                ->join('product_categories','product_categories.id','products.product_category_id')
                ->leftJoin('product_images', 'product_images.product_id', 'products.id')
                ->leftJoin('establishments', 'establishments.id', 'products.establishment_id')
                ->where('establishments.merchant_id', auth()->user()->account->merchant_id)
                ->latest()
                ->get();
        }

        return Datatables::of($products)
            ->addIndexColumn()
            ->addColumn('checkbox', function($product){
                $checkbox = '<input class="mt-3" name="product[]" value='.base64_encode($product->id).' type="checkbox">';
                return $checkbox;           
            })            
            ->addColumn('name', function($product){
                $name = ucwords($product->name);
                $photo = (isset($product->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$product->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('price', function($product){
                return '₱ '.number_format($product->price,2);
            })
            ->addColumn('status', function($product){
                if ($product->status == 0) {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending';
                } elseif($product->status == 1) {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Approved';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Rejected';
                }
                return $status;
            })
            ->addColumn('created_at', function($product){
                return $product->created_at->format('M d,Y h:i:s A');
            })
            ->escapeColumns([])->make(true);
    }

    // get active product categories
    public function getActiveProductCategory()
    {
        return ProductCategory::whereStatus(1)->get(['id','name']);
    }

    // get active product categories
    public function getActiveMerchantEstablishment()
    {
        return Establishment::where('merchant_id', auth()->user()->account->merchant_id)->whereStatus(1)->get(['id','name']);
    }
}
