<?php

namespace App\Http\Controllers\Dashboard\Merchant\Patient;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\DoctorAppointment;
use App\Models\Member;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Doctor Appointment View',
            'reference_id' => null,
            'table' => 'doctor_appointments',
            'remarks' => 'Doctor Appointment View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.patients.index');
    }

    public function patientList()
    {
    	$patients = DoctorAppointment::select([
                    'user_images.path',
                    'users.id',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'members.mobile_number',
                    'doctor_appointments.status',
                    'doctor_appointments.created_at',
                ])
            	->leftJoin('doctors', 'doctors.id', 'doctor_appointments.doctor_id')
                ->leftJoin('users', 'users.id', 'doctor_appointments.user_id')
            	->leftJoin('members', 'members.user_id', 'users.id')
                ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                ->where('doctors.merchant_id', auth()->user()->account->merchant_id)
                ->groupBy('user_images.path','users.id','users.firstname','users.middlename','users.lastname','users.email','members.mobile_number','doctor_appointments.status','doctor_appointments.created_at')
            	->latest()
                ->get();

        return Datatables::of($patients)
            ->addIndexColumn()
            ->addColumn('name', function($patient){
                $name = ucwords($patient->firstname.' '.$patient->middlename.' '.$patient->lastname);
                $photo = (isset($patient->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$patient->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('email', function($patient){
                return $patient->email;
            })
            ->addColumn('mobile_number', function($patient){
                return $patient->mobile_number;
            })
            ->addColumn('created_at', function($patient){
                return $patient->created_at->format('M d,Y h:i:s A');
            })
            ->addColumn('action', function($patient){
                $button = '<a href='.route('merchants.patients.view', base64_encode($patient->id)).' id="view">
                                <button class="btn btn-sm btn-info mr-1"> 
                                    <i class="mdi mdi-eye"></i> View
                                </button>
                            </a>';
                $action = '<div style="display:flex">
                                '.$button.'
                           </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function view($id)
    {
        $member = Member::select([
                    'user_images.path',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'members.*',
                ])
                ->join('users', 'users.id', 'members.user_id')
                ->leftJoin('user_images', 'user_images.auth_by', 'members.user_id')
                ->where('members.user_id', base64_decode($id))
                ->first();

        $member['path'] = isset($member->path) ? config('app.AWS_BUCKET_URL').$member->path : asset('admin_dashboard/img/default.png');

        return $member;
    }

    public function getPatientAppointments($user_id)
    {
        $appointments = DoctorAppointment::select([
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'doctors.professional_id',
                    'doctor_appointments.date_time',
                    'doctor_appointments.status',
                ])
                ->leftJoin('doctors', 'doctors.id', 'doctor_appointments.doctor_id')
                ->leftJoin('users', 'users.id', 'doctor_appointments.user_id')
                ->where('doctor_appointments.user_id', $user_id)
                ->get();

        return Datatables::of($appointments)
            ->addIndexColumn()
            ->addColumn('name', function($appointment){
                $name = ucwords($appointment->firstname.' '.$appointment->middlename.' '.$appointment->lastname);
                $photo = (isset($appointment->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$appointment->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('date_time', function($appointment){
                return $appointment->date_time;
            })  
            ->addColumn('status', function($appointment){
                if ($appointment->status == 'pending') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending' ;
                } elseif ($appointment->status == 'approved') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-info"></i> Approved';
                } elseif ($appointment->status == 'completed'){
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Completed';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> '.ucwords($appointment->status);
                }
                return $status;
            })            
            ->escapeColumns([])->make(true);
    }
}
