<?php

namespace App\Http\Controllers\Dashboard\Merchant\Order;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\TransactionOrder;
use App\Models\Driver;
use App\Models\Order;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Orders View',
            'reference_id' => null,
            'table' => 'orders',
            'remarks' => 'Orders View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.orders.index');
    }

    public function orderList(Request $request)
    {   
    	$transactions = $this->customStatusTransaction($request);

        return Datatables::of($transactions)
            ->addIndexColumn()
            ->addColumn('reference_id', function($transaction){
                return $transaction->reference_id;
            })
            ->addColumn('name', function($transaction){
                $name = ucwords($transaction->firstname.' '.$transaction->middlename.' '.$transaction->lastname);
                $photo = (isset($transaction->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$transaction->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('address', function($transaction){
                return $transaction->address;
            })
            ->addColumn('date_scheduled', function($transaction){
                return $transaction->updated_at;
            })
            ->addColumn('status', function($transaction){
                if ($transaction->status == 'pending') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending' ;
                } elseif ($transaction->status == 'pending_customer') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending Customer';
                } elseif ($transaction->status == 'accepted' || $transaction->status == 'prepared'){
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-info"></i> '.ucwords($transaction->status);
                } elseif ($transaction->status == 'cancelled') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Cancelled';
                } elseif ($transaction->status == 'otw') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-primary"></i> On The Way';
                } elseif ($transaction->status == 'arrived') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-secondary"></i> Arrived';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Completed';
                }
                return $status;
            })
            ->addColumn('action', function($transaction){
                $button = '<a href='.route('merchants.orders.view',base64_encode($transaction->id)).' id="view">
                                <button class="btn btn-sm btn-info mr-1"> 
                                    <i class="mdi mdi-eye"></i> View Orders
                                </button>
                            </a>';
                $action = '<div style="display:flex">
                                '.$button.'
                           </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function customStatusTransaction($request)
    {
        if (!is_null($request->status)) {
            if ($request->status == 'ongoing') {

                $transactions = TransactionOrder::select([
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'user_images.path',
                    'transaction_orders.id',
                    'transaction_orders.establishment_id',
                    'transaction_orders.reference_id',
                    'transaction_orders.address',
                    'transaction_orders.status',
                    'transaction_orders.created_at',
                    'transaction_orders.updated_at',
                ])
                ->leftJoin('users', 'users.id', 'transaction_orders.user_id')
                ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                ->where('transaction_orders.establishment_id', auth()->user()->account->merchant->merchantAgreement->establishment_id)
                ->whereNotIn('transaction_orders.status', ['pending', 'cancelled', 'completed'])
                ->latest()
                ->get();

            } else {

                $transactions = TransactionOrder::select([
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'user_images.path',
                    'transaction_orders.id',
                    'transaction_orders.establishment_id',
                    'transaction_orders.reference_id',
                    'transaction_orders.address',
                    'transaction_orders.status',
                    'transaction_orders.created_at',
                    'transaction_orders.updated_at',
                ])
                ->leftJoin('users', 'users.id', 'transaction_orders.user_id')
                ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                ->where('transaction_orders.establishment_id', auth()->user()->account->merchant->merchantAgreement->establishment_id)
                ->where('transaction_orders.status', $request->status)
                ->latest()
                ->get();
            }
        } else {

            $transactions = TransactionOrder::select([
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'user_images.path',
                    'transaction_orders.id',
                    'transaction_orders.establishment_id',
                    'transaction_orders.reference_id',
                    'transaction_orders.address',
                    'transaction_orders.status',
                    'transaction_orders.created_at',
                    'transaction_orders.updated_at',
                ])
                ->leftJoin('users', 'users.id', 'transaction_orders.user_id')
                ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                ->where('transaction_orders.establishment_id', auth()->user()->account->merchant->merchantAgreement->establishment_id)
                ->latest()
                ->get();
        }

        return $transactions;
    }

    public function view($id)
    {
        $transaction = TransactionOrder::select([
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'members.mobile_number',
                    'members.address as customer_address',
                    'transaction_orders.id',
                    'transaction_orders.reference_id',
                    'transaction_orders.driver_id',
                    'transaction_orders.address',
                    'transaction_orders.payment_method',
                    'transaction_orders.cash_on_hand',
                    'transaction_orders.delivery_amount',
                    'transaction_orders.notes',
                    'transaction_orders.status',
                    'transaction_orders.created_at',
                    'transaction_orders.updated_at',
                ])
                ->leftJoin('orders', 'orders.transaction_order_id', 'transaction_orders.id')
                ->leftJoin('users', 'users.id', 'transaction_orders.user_id')
                ->leftJoin('members', 'members.user_id', 'users.id')
                ->leftJoin('products', 'products.id', 'orders.product_id')
                ->where('transaction_orders.id', base64_decode($id))
                ->first();

        $transaction['name'] = ucwords($transaction->firstname.' '.$transaction->middlename.' '.$transaction->lastname);
        $transaction['date_ordered'] = $transaction->created_at->format('M d, Y h:i:s A');
        $transaction['date_scheduled'] = $transaction->updated_at->format('M d, Y h:i:s A');
        $transaction['drivers'] = $this->getMerchantActiveDrivers();
        if($transaction->status != 'cancelled') {
            $orders = Order::where('transaction_order_id', base64_decode($id))->count();
            if($orders == 0) {

                $transaction = TransactionOrder::find(base64_decode($id));
                $transaction->status = 'cancelled';
                $transaction->save();
                return response()->json(['message' => 'No Orders']);
            }
        }
        return $transaction;
    }

    public function getCustomerOrders($transaction_id)
    {
        $orders = Order::select([
                'product_images.path as image',
                'products.name',
                'products.price',
                'orders.quantity',
                'orders.id'
            ])
            ->join('products','products.id','orders.product_id')
            ->leftJoin('product_images','product_images.product_id','products.id')
            ->where('orders.transaction_order_id', $transaction_id)
            ->get();

        $transaction = TransactionOrder::find($transaction_id);

        return Datatables::of($orders)
            ->addIndexColumn()
            ->addColumn('image', function($order){
                $image = (isset($order->image)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$order->image.'" class="shadow" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="shadow" width="40px" height="40px">' ;
                return $image;
            })
            ->addColumn('name', function($order){
                return $order->name;
            })
            ->addColumn('price', function($order){
                return '₱'. number_format($order->price,2);
            })
            ->addColumn('total_price', function($order){
                $total_price = $order->price * $order->quantity;
                return '₱'. number_format($total_price,2);
            })
            ->addColumn('quantity', function($order){
                return $order->quantity;
            })
            ->addColumn('action', function($order) use($transaction){
                if($transaction->status == 'pending') {
                    $button = '<a href='.route('merchants.order.update.qty',base64_encode($order->id)).' id="edit">
                                    <button class="btn btn-sm btn-info mr-1"> 
                                        <i class="mdi mdi-pencil"></i>
                                    </button>
                                </a>';
                    $button .= '<a href='.route('merchants.order.update.status',base64_encode($order->id)).' id="delete">
                                    <button class="btn btn-sm btn-danger mr-1"> 
                                        <i class="mdi mdi-delete"></i>
                                    </button>
                                </a>';
                    $action = '<div style="display:flex">
                                    '.$button.'
                               </div>';
                    return $action;
                }
                
                return '';
            })            
            ->escapeColumns([])->make(true);
    }

    public function getMerchantActiveDrivers()
    {
        $drivers = Driver::select([
                    'drivers.id',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'drivers.created_at',
                ])
                ->leftJoin('users', 'users.id', 'drivers.user_id')
                ->where('drivers.merchant_id', auth()->user()->account->merchant_id)
                ->where('users.status', 1)
                ->get();

        return $drivers;
    }
}
