<?php

namespace App\Http\Controllers\Dashboard\Merchant\Driver;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\Merchant;
use App\Models\Vehicle;
use App\Models\Driver;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Driver View',
            'reference_id' => null,
            'table' => 'drivers',
            'remarks' => 'Driver View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.drivers.index');
    }

    public function driverList()
    {   
        if (auth()->user()->isSuperAdmin() || auth()->user()->isAdmin()) {
            $drivers = Driver::select([
                    'users.id as user_id',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'users.status',
                    'drivers.id',
                    'drivers.merchant_id',
                    'drivers.contact_number',
                    'vehicles.name as vehicle',
                    'drivers.created_at',
                ])
                ->leftJoin('users', 'users.id', 'drivers.user_id')
                ->leftJoin('vehicles', 'vehicles.id', 'drivers.vehicle_id')
                ->latest()
                ->get();
        } else {
            $drivers = Driver::select([
                    'users.id as user_id',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'users.status',
                    'drivers.id',
                    'drivers.merchant_id',
                    'drivers.contact_number',
                    'vehicles.name as vehicle',
                    'drivers.created_at',
                ])
                ->leftJoin('users', 'users.id', 'drivers.user_id')
                ->leftJoin('vehicles', 'vehicles.id', 'drivers.vehicle_id')
                ->where('drivers.merchant_id', auth()->user()->account->merchant_id)
                ->latest()
                ->get();
        }
    	
        return Datatables::of($drivers)
            ->addIndexColumn()
            ->addColumn('name', function($driver){
                $name = ucwords($driver->firstname.' '.$driver->middlename.' '.$driver->lastname);
                $photo = (isset($driver->image)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$driver->image->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('email', function($driver){
                return $driver->email;
            })
            ->addColumn('contact_number', function($driver){
                return $driver->contact_number;
            })
            ->addColumn('vehicle', function($driver){
                return $driver->vehicle;
            })
            ->addColumn('status', function($driver){
                if (auth()->user()->isSuperAdmin()) {
                    if ($driver->status == 1) {
                        $status = isset($driver->merchant_id) ? '<i class="mdi mdi-checkbox-blank-circle text-info"></i> Taken' : '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Available';
                    } else {
                         $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Inactive';
                    }
                } else {
                    $status = $driver->status == 1 ? 
                        '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active' :
                        '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Inactive' ;
                }
                return $status;
            })
            ->addColumn('action', function ($driver){
                $status = $driver->status == 1 ? 'Deactivate' : 'Activate';
                $icon = $driver->status == 1 ? '<i class="mdi mdi-minus-circle-outline"></i> Deactivate' : 
                        '<i class="mdi mdi-check-circle-outline"></i> Activate';
                $button = '<a href='.route('merchants.drivers.edit',base64_encode($driver->user_id)).' class="dropdown-item" id="edit">
                                <i class="ti-pencil"></i> Edit
                            </a>';
                $button .= '<a href='.route('merchants.drivers.status',base64_encode($driver->user_id)).' class="dropdown-item" id="status" data-status='.$status.'>
                                    '.$icon.'
                                </a>' ;
                $button .= '<div class="dropdown-divider"></div>';
                $button .= '<a href='.route('merchants.drivers.reset',base64_encode($driver->user_id)).' class="dropdown-item" id="resetPassword">
                                <i class="ti-lock"></i> Reset Password
                            </a>';
                $action = '<div class="btn-group">
                                <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    <i class="ti-view-list-alt"></i> Options
                                </button>
                                <div class="dropdown-menu dropdown-menu-right">
                                    '.$button.'
                                </div>
                            </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function add()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Drivers Add',
            'reference_id' => null,
            'table' => 'users',
            'remarks' => 'Successfully viewed Add driver page.',
        ];
        
        $this->addLogs($params);

        return view('dashboard.merchants.drivers.add',[
            'vehicles' => $this->getActiveVehicles(),
            'merchants' => $this->getAllMerchant(),
        ]);
    }

    public function edit($id)
    {
        $driver = Driver::select([
                    'users.id as user_id',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'users.username',
                    'users.status',
                    'drivers.id',
                    'drivers.merchant_id',
                    'drivers.contact_number',
                    'drivers.created_at',
                    'drivers.vehicle_id',
                ])
                ->leftJoin('users', 'users.id', 'drivers.user_id')
                ->leftJoin('vehicles', 'vehicles.id', 'drivers.vehicle_id')
                ->where('drivers.user_id', base64_decode($id))
                ->first();

        $driver['path'] = isset($driver->image) ? config('app.AWS_BUCKET_URL').$driver->image->path : '' ;

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchant Driver Edit',
            'reference_id' => $driver->id,
            'table' => 'users, drivers and vehicles',
            'remarks' => 'Successfully viewed Edit merchant driver page.',
        ];
        
        $this->addLogs($params);

        return view('dashboard.merchants.drivers.edit',[
            'driver'  => $driver,
            'vehicles' => $this->getActiveVehicles($driver->vehicle_id),
            'merchants' => $this->getAllMerchant(),
        ]);
    }

    // get active vehicles
    public function getActiveVehicles($id = null)
    {   
        if ($id) {
            return Vehicle::whereStatus(1)->orWhere('id', $id)->get(['id','name']);
        } 
        return Vehicle::whereStatus(1)->get(['id','name']);
    }

    // get all merchant
    public function getAllMerchant()
    {
        return Merchant::orderBy('name')->get(['id','name']);
    }
}
