<?php

namespace App\Http\Controllers\Dashboard\Merchant\Doctor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\DoctorPaymentMethod;
use App\Models\DoctorSpeciality;
use App\Models\DoctorSchedule;
use App\Models\Speciality;
use App\Models\Doctor;
use App\Models\User;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Doctor View',
            'reference_id' => null,
            'table' => 'doctors',
            'remarks' => 'Doctor View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.doctors.index');
    }

    public function merchantDoctorList()
    {
    	$doctors = Doctor::select([
                        'doctors.id',
                        'doctors.professional_id',
                        'doctor_schedules.doctor_id',
                        'users.id as user_id',
                        'users.firstname',
                        'users.middlename',
                        'users.lastname',
                        'users.email',
                        'users.status',
	                ])
	            	->leftJoin('users', 'doctors.user_id', 'users.id')
                    ->leftJoin('doctor_schedules', 'doctor_schedules.doctor_id', 'doctors.id')
	            	->where('doctors.merchant_id', auth()->user()->account->merchant_id)
	                ->get();

        return Datatables::of($doctors)
            ->addIndexColumn()
            ->addColumn('name', function($doctor){
                $name = ucwords($doctor->firstname.' '.$doctor->middlename.' '.$doctor->lastname);
                $photo = (isset($doctor->image->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$doctor->image->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('professional_id', function($doctor){
                return $doctor->professional_id;
            })
            ->addColumn('specialities', function($doctor){
                return  $this->getDoctorSpecialitiesName($doctor);
            })
            ->addColumn('status', function($doctor){
                $status = $doctor->status == 1 ? 
                    '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active' :
                    '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Not Active' ;
                return $status;
            })
            ->addColumn('action', function ($doctor){
                $status = $doctor->status == 1 ? 'Deactivate' : 'Activate';
                $icon = $doctor->status == 1 ? '<i class="mdi mdi-minus-circle-outline"></i> Deactivate' : 
                        '<i class="mdi mdi-check-circle-outline"></i> Activate';
                $button = '<a href='.route('merchants.doctors.view',base64_encode($doctor->user_id)).' class="dropdown-item" id="view">
                                <i class="ti-eye"></i> View
                            </a>';
                $button .= '<a href='.route('merchants.doctors.edit',base64_encode($doctor->user_id)).' class="dropdown-item" id="edit">
                                <i class="ti-pencil"></i> Edit
                            </a>';
                $button .= '<a href='.route('merchants.doctors.status',base64_encode($doctor->user_id)).' class="dropdown-item" id="status" data-status='.$status.'>
                                '.$icon.'
                            </a>' ;
                $button .= '<div class="dropdown-divider"></div>';
                if ($doctor->doctor_id != null) {
                    $button .= '<a href='.route('merchants.doctors.schedules.edit',base64_encode($doctor->id)).' class="dropdown-item">
                                <i class="ti-calendar"></i> Edit Schedule
                            </a>';
                } else {
                    $button .= '<a href='.route('merchants.doctors.schedules.add',base64_encode($doctor->id)).' class="dropdown-item">
                                <i class="ti-calendar"></i> Add Schedule
                            </a>';
                }
                
                $action = '<div class="btn-group">
                                <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    <i class="ti-view-list-alt"></i> Options
                                </button>
                                <div class="dropdown-menu dropdown-menu-right">
                                    '.$button.'
                                </div>
                            </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function add()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Doctor Add',
            'reference_id' => null,
            'table' => 'doctors',
            'remarks' => 'Successfully viewed Add doctor page.',
        ];
        
        $this->addLogs($params);

        return view('dashboard.merchants.doctors.add',[
            'specialities' => $this->getActiveSpecialities()
        ]);
    }

    public function edit($id)
    {   
        $doctor = Doctor::select([
                    'doctors.id',
                    'doctors.prefix',
                    'doctors.contact_number',
                    'doctors.minimum_consultation_fee',
                    'doctors.maximum_consultation_fee',
                    'users.id as user_id',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'users.username',
                ])
                ->leftJoin('users', 'doctors.user_id', 'users.id')
                ->where('doctors.user_id', base64_decode($id))
                ->first();

        $doctor['path'] = isset($doctor->image) ? config('app.AWS_BUCKET_URL').$doctor->image->path : '' ;
        $doctor['payment_methods'] = $this->getDoctorPaymentMethod($doctor->id);
        $doctor['specialities'] = $this->getDoctorSpecialities($doctor->id);

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Doctor Edit',
            'reference_id' => $doctor->id,
            'table' => 'users and doctors',
            'remarks' => 'Successfully viewed Edit doctor page.',
        ];
        
        $this->addLogs($params);

        return view('dashboard.merchants.doctors.edit',[
            'specialities' => $this->getActiveSpecialities(),
            'doctor'         => $doctor,
        ]);
    }

    public function view($id)
    {
        $doctor = Doctor::select([
                    'doctors.id',
                    'doctors.prefix',
                    'doctors.professional_id',
                    'doctors.contact_number',
                    'doctors.minimum_consultation_fee',
                    'doctors.maximum_consultation_fee',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'users.username',
                    'users.usertype',
                    'users.status',
                ])
                ->leftJoin('users', 'users.id', 'doctors.user_id')
                ->where('doctors.user_id', base64_decode($id))
                ->first();

        $doctor['name'] = ucwords($doctor->firstname.' '.$doctor->middlename.' '.$doctor->lastname);
        $doctor['path'] = isset($doctor->image) ? config('app.AWS_BUCKET_URL').$doctor->image->path : asset("admin_dashboard/img/default.png");
        $doctor['payment_methods'] = $this->getDoctorPaymentMethod($doctor->id);
        $doctor['specialities'] = $this->getDoctorSpecialitiesName($doctor);
        $doctor['schedules'] = $this->getDoctorSchedules($doctor->id);

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Doctor View',
            'reference_id' => $doctor->id,
            'table' => 'users and doctors',
            'remarks' => 'Successfully viewed doctor.',
        ];
        
        $this->addLogs($params);

        return $doctor;
    }

    public function getActiveSpecialities()
    {
        return Speciality::whereStatus(1)->get(['id','name']);
    }

    public function getDoctorPaymentMethod($id)
    {
        $paymentMethods = DoctorPaymentMethod::where('doctor_id', $id)->get();

        $data = [];
        foreach ($paymentMethods as $payment) {
            array_push($data, $payment->name);
        }
        
        return $data;
    }

    public function getDoctorSpecialities($id)
    {
        $specialities = DoctorSpeciality::where('doctor_id', $id)->get();

        $data = [];
        foreach ($specialities as $doctor) {
            array_push($data, $doctor->speciality_id);
        }
        
        return $data;
    }

    public function getDoctorSpecialitiesName($doctor)
    {
        $data = [];
        foreach ($doctor->doctorSpecialities as $doctor) {
            $data[] = $doctor->speciality->name;
        }
        return implode(', ', $data);
    }

    public function getDoctorSchedules($doctor_id)
    {
        $schedules = DoctorSchedule::where('doctor_id', $doctor_id)->first();
 
        return $schedules ? $schedules : $schedules = [];
    }
}
