<?php

namespace App\Http\Controllers\Dashboard\Merchant\Doctor\Appointment;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\DoctorPaymentMethod;
use App\Models\DoctorAppointment;

class ViewController extends Controller
{
    public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Doctor Appointment View',
            'reference_id' => null,
            'table' => 'doctor_appointments',
            'remarks' => 'Doctor Appointment View Success',
        ];

        $this->addLogs($params);
        
    	return view('dashboard.merchants.doctors.appointments.index');
    }

    public function doctorAppointmentList(Request $request)
    {
    	$appointments = $this->customStatusAppointment($request);

        return Datatables::of($appointments)
            ->addIndexColumn()
            ->addColumn('name', function($appointment){
                $name = ucwords($appointment->firstname.' '.$appointment->middlename.' '.$appointment->lastname);
                $photo = (isset($appointment->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$appointment->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('reference_id', function($appointment){
                return $appointment->reference_id;
            })
            ->addColumn('date_time', function($appointment){
                return $appointment->date_time;
            })
            ->addColumn('payment_method', function($appointment){
            	return ucwords($appointment->payment_method);
            })
            ->addColumn('status', function($appointment){
                if ($appointment->status == 'pending') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending' ;
                } elseif ($appointment->status == 'approved') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-info"></i> Approved';
                } elseif ($appointment->status == 'completed'){
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Completed';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> '.ucwords($appointment->status);
                }
                return $status;
            })
            ->addColumn('action', function($appointment){
                $button = '<a href='.route('doctors.appointments.view', base64_encode($appointment->id)).' id="view">
                                <button class="btn btn-sm btn-info mr-1"> 
                                    <i class="mdi mdi-eye"></i> View
                                </button>
                            </a>';
                $action = '<div style="display:flex">
                                '.$button.'
                           </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function customStatusAppointment($request)
    {
        if (!is_null($request->status)) {
            $appointments = DoctorAppointment::select([
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'user_images.path',
                    'doctor_appointments.id',
                    'doctor_appointments.reference_id',
                    'doctor_appointments.payment_method',
                    'doctor_appointments.date_time',
                    'doctor_appointments.status',
                    'doctor_appointments.created_at',
                ])
            	->leftJoin('doctors', 'doctors.user_id', 'doctor_appointments.doctor_id')
                ->leftJoin('users', 'users.id', 'doctor_appointments.user_id')
                ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                ->where('doctors.user_id', auth()->user()->id)
                ->where('doctor_appointments.status', $request->status)
            	->latest()
                ->get();
        } else {

            $appointments = DoctorAppointment::select([
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'user_images.path',
                    'doctor_appointments.id',
                    'doctor_appointments.reference_id',
                    'doctor_appointments.payment_method',
                    'doctor_appointments.date_time',
                    'doctor_appointments.status',
                    'doctor_appointments.created_at',
                ])
            	->leftJoin('doctors', 'doctors.user_id', 'doctor_appointments.doctor_id')
                ->leftJoin('users', 'users.id', 'doctor_appointments.user_id')
                ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                ->where('doctors.user_id', auth()->user()->id)
            	->latest()
                ->get();
        }

        return $appointments;
    }

    public function view($id)
    {
        $appointment = DoctorAppointment::select([
                    'user_images.path',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'members.mobile_number',
                    'members.birthdate',
                    'members.gender',
                    'members.marital_status',
                    'members.citizenship',
                    'members.citizenship',
                    'doctor_appointments.*',
                ])
                ->leftJoin('doctors', 'doctors.id', 'doctor_appointments.doctor_id')
                ->leftJoin('users', 'users.id', 'doctor_appointments.user_id')
                ->leftJoin('members', 'members.user_id', 'users.id')                
                ->leftJoin('user_images', 'user_images.auth_by', 'users.id')
                ->where('doctor_appointments.id', base64_decode($id))
                ->first();

        $appointment['path'] = isset($appointment->path) ? config('app.AWS_BUCKET_URL').$appointment->path :
             asset("admin_dashboard/img/default.png");
        $appointment['name'] = ucwords($appointment->firstname.' '.$appointment->middlename.' '.$appointment->lastname);
        $appointment['date_requested'] = $appointment->created_at->format('M d, Y h:i:s A');
        $appointment['date_updated'] = $appointment->updated_at->format('M d, Y h:i:s A');
        $appointment['hmo'] = $this->getPaymentMethod($appointment->doctor_id);

        return $appointment;
    }

    // get payment methods or hmo
    public function getPaymentMethod($id)
    {
        $paymentMethods = DoctorPaymentMethod::where('doctor_id', $id)->get(['name']);

        $data = [];
        foreach ($paymentMethods as $hmo) {
            $data[] = str_replace('_',' ', $hmo->name);
        }
        return implode(', ', $data);
    }
}
