<?php

namespace App\Http\Controllers\Dashboard\Merchant\Customer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\TransactionOrder;
use App\Models\Member;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Customer View',
            'reference_id' => null,
            'table' => 'customers',
            'remarks' => 'Customer View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.customers.index');
    }

    public function customerList()
    {   
        $customers = TransactionOrder::select([
                        'users.id as user_id',
                        'user_images.path',
                        'users.firstname',
                        'users.middlename',
                        'users.lastname',
                        'users.status',
                        'members.member_id',
                        'members.address',
                        'members.registration_status',
                        'transaction_orders.id as transaction_id',
                    ])
                    ->join('users', 'users.id', 'transaction_orders.user_id')
                    ->leftJoin('members', 'members.user_id', 'users.id')
                    ->leftJoin('user_images','user_images.auth_by','users.id')
                    ->where('transaction_orders.establishment_id', auth()->user()->account->merchant->merchantAgreement->establishment_id)
                    ->groupBy('transaction_orders.user_id','users.id','user_images.path','users.firstname','users.middlename','users.lastname','users.status','members.address','members.registration_status','transaction_orders.id','members.member_id')
                    ->get();

        return Datatables::of($customers)
            ->addIndexColumn()
            ->addColumn('member_id', function($customer){
                return $customer->member_id;
            })
            ->addColumn('name', function($customer){
                $name = ucwords($customer->firstname.' '.$customer->middlename.' '.$customer->lastname);
                $photo = (isset($customer->path)) ? 
                            '<img src="'.config('app.AWS_BUCKET_URL').$customer->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                            '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;
                return $photo.' '.$name;
            })
            ->addColumn('address', function($customer){
                return $customer->address;
            })
            ->addColumn('member_status', function($customer){
                return ($customer->registration_status == 'unverified') ? 'NON-MEMBER' : 'MEMBER';
            })
            ->addColumn('status', function($customer){
            	$status = $customer->status == 1 ? 
                    '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active' :
                    '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Not Active' ;
                return $status;
            })
            ->addColumn('action', function ($customer){
                $button = '<a href='.route('merchants.customers.view',base64_encode($customer->user_id)).' id="view">
                                <button class="btn btn-sm btn-info mr-1"> 
                                    <i class="mdi mdi-eye"></i> View
                                </button>
                            </a>';
                $button .= '<a href='.route('merchants.customers.transactions',base64_encode($customer->transaction_id)).' id="transactions">
                                <button class="btn btn-sm btn-primary mr-1"> 
                                    <i class="mdi mdi-file-document-box"></i> Transactions
                                </button>
                            </a>';
                $action = '<div style="display:flex">
                                '.$button.'
                           </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function view($id)
    {
        $member = Member::select([
                    'user_images.path',
                    'users.firstname',
                    'users.middlename',
                    'users.lastname',
                    'users.email',
                    'members.address',
                    'members.birthdate',
                    'members.mobile_number',
                    'members.citizenship',
                    'members.marital_status',
                    'members.birthdate',
                    'members.gender',
                    'members.registration_status',
                ])
                ->join('users','users.id','members.user_id')
                ->leftJoin('user_images','user_images.auth_by','members.user_id')
                ->where('members.user_id', base64_decode($id))
                ->first();

        $member['path'] = isset($member->path) ? config('app.AWS_BUCKET_URL').$member->path : 'admin_dashboard/img/default.png';
        $member['name'] = ucwords($member->firstname.' '.$member->middlename.' '.$member->lastname);
        $member['birth_date'] = $member->birthdate->format('M d, Y');

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Customer View',
            'reference_id' => $member->id,
            'table' => 'users and members',
            'remarks' => 'Successfully viewed customer.',
        ];
        
        $this->addLogs($params);

        return $member;
    }

    public function transactions($transaction_id)
    {
        $transactions = TransactionOrder::select([
                    'reference_id',
                    'status',
                    'created_at',
                    'updated_at',
                ])
                ->where('id', base64_decode($transaction_id))
                ->latest()
                ->get();

        return Datatables::of($transactions)
            ->addIndexColumn()
            ->addColumn('reference_id', function($transaction){
                return $transaction->reference_id;
            })
            ->addColumn('updated_at', function($transaction){
                return $transaction->updated_at->format('M d,Y h:i:s A');
            })
            ->addColumn('status', function($transaction){
                if ($transaction->status == 'pending') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-warning"></i> Pending' ;
                } elseif ($transaction->status == 'accepted' || $transaction->status == 'prepared'){
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-info"></i> '.ucwords($transaction->status);
                } elseif ($transaction->status == 'cancelled') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Cancelled';
                } elseif ($transaction->status == 'otw') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-primary"></i> On The Way';
                } elseif ($transaction->status == 'arrived') {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-secondary"></i> Arrived';
                } else {
                    $status = '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Completed';
                }
                return $status;
            })         
            ->escapeColumns([])->make(true);
    }
}
