<?php

namespace App\Http\Controllers\Dashboard\Merchant\Cashier;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\User;

class ViewController extends Controller
{
    public function index()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Cashier View',
            'reference_id' => null,
            'table' => 'cashiers',
            'remarks' => 'Cashier View Success',
        ];

        $this->addLogs($params);

    	return view('dashboard.merchants.cashiers.index');
    }

    public function merchantCashierList()
    {
    	$users = User::select([
    					'users.id',
    					'users.firstname',
    					'users.middlename',
    					'users.lastname',
    					'users.email',
    					'users.status',
    					'merchant_cashiers.cashier_id',
    					'merchant_cashiers.employee_number',
    					'merchant_cashiers.department',
    					'merchant_cashiers.created_at',
    				])
    				->join('merchant_cashiers','users.id','merchant_cashiers.user_id')
                    ->where('users.usertype','merchant_cashier')
                    ->where('merchant_cashiers.merchant_id', auth()->user()->account->merchant_id)
                    ->latest()
                    ->get();

        return Datatables::of($users)
            ->addIndexColumn()
            ->addColumn('name', function($user){
                $name = ucwords($user->firstname.' '.$user->middlename.' '.$user->lastname);
                $photo = (isset($user->images[0]->path)) ? 
                        '<img src="'.config('app.AWS_BUCKET_URL').$user->images[0]->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                        '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;
            })
            ->addColumn('email', function($user){
                return $user->email;
            })
            ->addColumn('department', function($user){
                return strtoupper($user->department);
            })
            ->addColumn('employee_number', function($user){
                return $user->employee_number;
            })
            ->addColumn('status', function($user){
                $status = $user->status == 1 ? 
                    '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active' :
                    '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Not Active' ;
                return $status;
            })
            ->addColumn('action', function ($user){
                $status = $user->status == 1 ? 'Deactivate' : 'Activate';
                $icon = $user->status == 1 ? '<i class="mdi mdi-minus-circle-outline"></i> Deactivate' : 
                        '<i class="mdi mdi-check-circle-outline"></i> Activate';
                $button = '<a href='.route('merchants.cashiers.view',base64_encode($user->id)).' class="dropdown-item" id="view">
                                <i class="ti-eye"></i> View
                            </a>';
                $button .= '<a href='.route('merchants.cashiers.edit',base64_encode($user->id)).' class="dropdown-item" id="edit">
                                <i class="ti-pencil"></i> Edit
                            </a>';
                $button .= '<a href='.route('merchants.cashiers.status',base64_encode($user->id)).' class="dropdown-item" id="status" data-status='.$status.'>
                                '.$icon.'
                            </a>' ;
                $button .= '<div class="dropdown-divider"></div>';
                $button .= '<a href='.route('merchants.cashiers.reset',base64_encode($user->id)).' class="dropdown-item" id="resetPassword">
                                <i class="ti-lock"></i> Reset Password
                            </a>';
            
                $action = '<div class="btn-group">
                                <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    <i class="ti-view-list-alt"></i> Options
                                </button>
                                <div class="dropdown-menu dropdown-menu-right">
                                    '.$button.'
                                </div>
                            </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function add()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Cashier Add',
            'reference_id' => null,
            'table' => 'users',
            'remarks' => 'Successfully viewed Add user page.',
        ];
        
        $this->addLogs($params);

    	return view('dashboard.merchants.cashiers.add');
    }

    public function edit($id)
    {
    	$user = User::select([
					'users.id',
					'users.firstname',
					'users.middlename',
					'users.lastname',
					'users.email',
					'users.username',
					'merchant_cashiers.employee_number',
					'merchant_cashiers.department',
				])
				->join('merchant_cashiers','users.id','merchant_cashiers.user_id')
                ->where('users.id', base64_decode($id))
                ->first();

        $user['path'] = isset($user->images[0]) ? config('app.AWS_BUCKET_URL').$user->images[0]->path : '' ;

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchant Cashier Edit',
            'reference_id' => $user->id,
            'table' => 'users',
            'remarks' => 'Successfully viewed Edit merchant cashier page.',
        ];
        
        $this->addLogs($params);

        return view('dashboard.merchants.cashiers.edit',[
            'user'  => $user,
        ]);
    }

    public function view($id)
    {
        $user = User::select([
					'users.id',
					'users.firstname',
					'users.middlename',
					'users.lastname',
					'users.email',
					'users.username',
					'users.usertype',
					'users.status',
					'merchant_cashiers.cashier_id',
					'merchant_cashiers.employee_number',
					'merchant_cashiers.department',
					'merchant_cashiers.created_at',
				])
				->join('merchant_cashiers','users.id','merchant_cashiers.user_id')
                ->where('users.id', base64_decode($id))
                ->first();

        $user['name'] = ucwords($user->firstname.' '.$user->middlename.' '.$user->lastname);
        $user['employee_number'] = $user->employee_number;
        $user['usertype'] = str_replace('_', ' ', $user->usertype);
        $user['department'] = $user->department;
        $user['date_registered'] = $user->created_at->format('M d,Y h:i:s A');
        $user['path'] = isset($user->images[0]) ? config('app.AWS_BUCKET_URL').$user->images[0]->path :
             asset("admin_dashboard/img/default.png");        

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Merchant Cashier View',
            'reference_id' => $user->id,
            'table' => 'users and merchant_cashiers',
            'remarks' => 'Successfully viewed merchant cashier.',
        ];
        
        $this->addLogs($params);

        return $user;
    }
}
