<?php

namespace App\Http\Controllers\Dashboard\Administration;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

//Add External Classes
use Yajra\Datatables\Datatables;

//Add Models
use App\Models\MerchantCashier;
use App\Models\Merchant;
use App\Models\User;
use App\Models\Log;

class ViewController extends Controller
{
    public function index()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Registered Users',
            'reference_id' => null,
            'table' => 'users',
            'remarks' => 'Success viewing registered users.',
        ];

        $this->addLogs($params);

    	return view('dashboard.administration.users.index');
    }

    public function userList()
    {   
        $users = User::whereNotIn('users.usertype', ['member','merchant','doctor','driver'])
                    ->where('users.id','!=','1')
                    ->latest()
                    ->get();

        return Datatables::of($users)
            ->addIndexColumn()
            ->addColumn('name', function($user){
                $name = $user->getFullName();
                $photo = (isset($user->images[0]->path)) ? 
                        '<img src="'.config('app.AWS_BUCKET_URL').$user->images[0]->path.'" class="rounded-circle mr-2" width="40px" height="40px">' : 
                        '<img src='.asset('admin_dashboard/img/default.png').' class="rounded-circle mr-2" width="40px" height="40px">' ;

                return $photo.' '.$name;
            })
            ->addColumn('company', function($user){
                return $this->getMerchantCompany($user);
            })
            ->addColumn('department', function($user){
                return strtoupper($user->account->department);
            })
            ->addColumn('usertype', function($user){

                return str_replace('_', '-', strtoupper($user->usertype));
            })
            ->addColumn('cashierid', function($user){

                $cashier = MerchantCashier::where('user_id',$user->id)
                         ->first();


                if($cashier!=null)
                {
                    return strtoupper($cashier->cashier_id);

                }
                else
                {
                    return "";
                }
            })
            ->addColumn('status', function($user){
                $status = $user->status == 1 ? 
                    '<i class="mdi mdi-checkbox-blank-circle text-success"></i> Active' :
                    '<i class="mdi mdi-checkbox-blank-circle text-danger"></i> Not Active' ;
                return $status;
            })
            ->addColumn('action', function ($user){
                $status = $user->status == 1 ? 'Deactivate' : 'Activate';
                $icon = $user->status == 1 ? '<i class="mdi mdi-minus-circle-outline"></i> Deactivate' : 
                        '<i class="mdi mdi-check-circle-outline"></i> Activate';
                $button = '<a href='.route('administration.users.view',base64_encode($user->id)).' class="dropdown-item" id="view">
                                <i class="ti-eye"></i> View
                            </a>';
                if (auth()->user()->isSuperAdmin() || auth()->user()->isAdmin()) {
                    $button .= '<a href='.route('administration.users.edit',base64_encode($user->id)).' class="dropdown-item" id="edit">
                                    <i class="ti-pencil"></i> Edit
                                </a>';
                    $button .= '<div class="dropdown-divider"></div>';
                    if(auth()->user()->isSuperAdmin()) {
                        $button .= '<a href='.route('administration.users.status',base64_encode($user->id)).' class="dropdown-item" id="status" data-status='.$status.'>
                                        '.$icon.'
                                    </a>' ;
                    }
                    $button .= '<a href='.route('administration.users.reset',base64_encode($user->id)).' class="dropdown-item" id="resetPassword">
                                    <i class="ti-lock"></i> Reset Password
                                </a>';
                }
                
                $action = '<div class="btn-group">
                                <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    <i class="ti-view-list-alt"></i> Options
                                </button>
                                <div class="dropdown-menu dropdown-menu-right">
                                    '.$button.'
                                </div>
                            </div>';
                return $action;
            })
            ->escapeColumns([])->make(true);
    }

    public function add()
    {   
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'User Add',
            'reference_id' => null,
            'table' => 'users',
            'remarks' => 'Successfully viewed Add user page.',
        ];
        
        $this->addLogs($params);

        return view('dashboard.administration.users.add');
    }

    public function edit($id)
    {   
        $user = User::find(base64_decode($id));
        $user['path'] = isset($user->images[0]) ? config('app.AWS_BUCKET_URL').$user->images[0]->path : '' ;

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'User Edit',
            'reference_id' => $user->id,
            'table' => 'users',
            'remarks' => 'Successfully viewed Edit user page.',
        ];
        
        $this->addLogs($params);

        return view('dashboard.administration.users.edit',[
            'user'  => $user,
        ]);
    }

    public function view($id)
    {
        $user = User::find(base64_decode($id));
        $user['name'] = $user->getFullName();
        $user['contact_number'] = $user->account->contact_number;
        $user['employee_number'] = $user->account->employee_number;
        $user['department'] = $user->account->department;
        $user['company'] = $this->getMerchantCompany($user);
        $user['usertype'] = str_replace('_', ' ', $user->usertype);
        $user['date_registered'] = $user->created_at->format('M d,Y h:i:s A');
        $user['path'] = isset($user->images[0]) ? config('app.AWS_BUCKET_URL').$user->images[0]->path :
             asset("admin_dashboard/img/default.png");        

        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'User View',
            'reference_id' => $user->id,
            'table' => 'users',
            'remarks' => 'Successfully viewed user.',
        ];
        
        $this->addLogs($params);

        return $user;
    }

    public function transactionLogs()
    {
        $params = [
            'user_id' => auth()->user()->id, 
            'activity' => 'Transaction Logs',
            'reference_id' => null,
            'table' => 'logs',
            'remarks' => 'Success viewing transaction logs.',
        ];

        $this->addLogs($params);

        return view('dashboard.administration.transactions.index');
    }

    public function transactionLogsList() 
    {
        $logs = Log::select([
                    'logs.*',
                    'users.firstname',
                    'users.lastname',
                ])
                ->join('users','users.id', 'logs.user_id')
                ->limit(1000)
                ->latest()
                ->get();

        return Datatables::of($logs)
            ->addColumn('name', function($log){
                return isset($log->firstname) ? $log->firstname.' '.$log->lastname : '';
            })
            ->addColumn('created_at', function($log){
                return $log->created_at->format('M d,Y h:i:s A');
            })
            ->escapeColumns([])->make(true);
    }

    public function getAllMerchant()
    {
        return Merchant::orderBy('name')->get(['id','name']);
    }

    // get merchant company
    public function getMerchantCompany($user)
    {
        if ($user->usertype == 'merchant_admin' || $user->usertype == 'merchant_cashier') {
            return Merchant::whereId($user->account->merchant_id)->value('name');
        } else {
            return 'PPLUS Operations and OFFICIAL PGB Shared Services 1';
        }
    }
}
