<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Watson\Validating\ValidatingTrait;
use Spatie\Permission\Traits\HasRoles;
use Laravel\Passport\HasApiTokens;
use App\Notifications\PasswordResetRequest;
use Illuminate\Contracts\Auth\CanResetPassword;

class User extends Authenticatable
{
    use Notifiable, HasApiTokens;
    use SoftDeletes;
    use ValidatingTrait;
    use HasRoles;
    
    protected $fillable = [
        'firstname', 
        'middlename', 
        'lastname', 
        'email', 
        'username', 
        'password',
        'usertype',
        'status',
    ];

    protected $dates = ['deleted_at'];

    protected $hidden = [
        'password', 
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    protected $rules = [
        'firstname' => 'required|regex:/^[a-zA-Z\s]*$/',
        'middlename'=> 'nullable|regex:/^[a-zA-Z\s]*$/',
        'lastname'  => 'required|regex:/^[a-zA-Z\s]*$/',
        'email'     => 'required|email|unique:users,email',
        'username'  => 'required|string|regex:/^\S*$/u|min:6|unique:users,username',
        'password'  => 'required|min:6',
        'usertype'  => 'required|in:super_admin,admin,human_resource,accounting,admin_cashier,cashier,merchant_admin,merchant_cashier,merchant,member',
    ];

    // Custom validation messages rules
    protected $validationMessages = [
        'username.regex' => "Username should not contain spaces"
    ];

    public static function rules()
    {
        return (new static)->rules;
    }
    
    public function images()
    {
        return $this->hasMany(UserImage::class, 'auth_by')->latest();
    }

    public function account()
    {   
        if ($this->usertype == 'admin_cashier' || $this->usertype == 'cashier') {
            return $this->hasOne(Cashier::class, 'user_id');
        } elseif ($this->usertype == 'merchant_admin') {
            return $this->hasOne(MerchantAccount::class, 'user_id');
        } elseif ($this->usertype == 'merchant_cashier') {
            return $this->hasOne(MerchantCashier::class, 'user_id');
        } else {
            return $this->hasOne(AdminAccount::class, 'user_id');
        }
    }

    public function isSuperAdmin()
    {
        return $this->hasRole('super_admin');
    }

    public function isAdmin()
    {
        return $this->hasRole('admin');
    }

    public function isMerchantAdmin()
    {
        return $this->hasRole('merchant_admin');
    }

    public function member()
    {
        return $this->hasOne(Member::class, 'user_id');
    }

    public function sendPasswordResetNotification($token)
    {
        $this->notify(new PasswordResetRequest($token));
    }
}
