<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Watson\Validating\ValidatingTrait;

class Merchant extends Model
{
    use ValidatingTrait;

    protected $fillable = [
        'merchant_type_id',
        'merchant_area_code_id',
        'name',
        'email',
        'contact_number',
        'contact_person',
        'address',
        'website',
        'mb_amount',
        'mb_percent',
        'deduction_on',
        'referral_code',
        'date_start',
        'date_end',
        'is_mobile',
    ];

    protected $dates = [
        'date_start',
        'date_end',
    ];
    
    protected $rules = [
        'image'                 => 'nullable|mimes:jpeg,jpg,png|max:10000',
        'merchant_type_id'      => 'required|integer',
        'merchant_area_code_id' => 'required|integer',
        'name'         		    => 'required|unique:merchants,name',
        'email'                 => 'required|email|unique:merchants,email',       
        'contact_number'        => 'required',
        'contact_person'        => 'required|regex:/^[a-zA-Z\s]*$/',       
        'address'               => 'required',   
        'website'    		    => 'nullable|regex:/^(https?:\/\/)?([\da-z\.-]+)\.([a-z\.]{2,6})([\/\w \.-]*)*\/?$/',       
        'mb_amount'    		    => 'nullable|integer',      
        'mb_percent'    	    => 'nullable|integer',      
        'deduction_on'    	    => 'nullable|in:points,rebate,system,accumulated,advocacy',      
        'referral_code'    	    => 'required|unique:merchants,referral_code',      
        'date_start'            => 'required|date',      
        'date_end'    		    => 'required|date',      
        'is_mobile'             => 'nullable|in:1,0',      
    ];

    public static function rules()
    {
        return (new static)->rules;
    }

    public function images()
    {
        return $this->hasMany(MerchantImage::class, 'merchant_id')->latest();
    }
    
    public function merchantAgreement()
    {
        return $this->hasOne(MerchantAgreement::class, 'merchant_id', 'id');
    }

    public function merchantWallets()
    {
        return $this->hasMany(MerchantWallet::class,'merchant_id', 'id');
    }

    public function points()
    {
        return $this->hasMany(Point::class, 'merchant_id');
    }

    public function establishments()
    {
        return $this->hasMany(Establishment::class,'merchant_id');
    }

    public function getWalletBalance()
    {
        if ($this->deduction_on == 'points') {
            return $this->merchantWallets->sum('amount') - $this->points->sum('amount');
        } elseif ($this->deduction_on == 'rebate') {
            return $this->merchantWallets->sum('amount') - $this->points->sum('rebate');
        } elseif ($this->deduction_on == 'system') {
            return $this->merchantWallets->sum('amount') - $this->points->sum('system');
        } elseif ($this->deduction_on == 'accumulated') {
            return $this->merchantWallets->sum('amount') - $this->points->sum('accumulated');
        } elseif ($this->deduction_on == 'advocacy') {
            return $this->merchantWallets->sum('amount') - $this->points->sum('advocacy');
        } else{
            return 0;
        }
    }
}
