<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;

//Add Storage Class
use Illuminate\Support\Facades\Storage;

//Add Role Class
use Spatie\Permission\Models\Role;

use App\Models\Member;
use App\Models\User;
use App\Models\Promotion;
use App\Models\PromotionLog;
use App\Models\PointImage;
use App\Models\Log;
use Carbon\Carbon;

class Controller extends BaseController
{
    use AuthorizesRequests, DispatchesJobs, ValidatesRequests;

    public function emitSocket($data) {
        $url = config('app.SOCKET_URL');

        $response = \Ixudra\Curl\Facades\Curl::to($url)
            ->withHeaders(array('Authorization' => 'test'))
            ->withData(json_encode($data, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE))->post();

        return $response;
    }

    //creating users
    public function assignRole(User $user) 
    {
        Role::findOrCreate($user->usertype);
        $user->assignRole($user->usertype);
    }

    public function getMemberStatus($member_id) {
    	$member = Member::where('user_id', $member_id)->first();
    	return $member;
    }

    public function getMember($user_id) {
        $member = Member::where('user_id', $user_id)->first();
        return $member;
    }

    public function updateReferralPoints($data, $user_id) {
        $member = Member::where('user_id', $user_id)->update($data);
        return $member;
    }

    public function successResponseAuth($message = null, $data, $token) {
        return response()->json([
            "status" => "SUCCESS",
            "message" => $message,
            "data" => $data
        ], 200,
        [
            'Content-Type' => 'application/json', 
            'Authorization' => (string)$token
        ]);
    }

    public function successResponse($message = null, $data) {
    	return response()->json([
    		"status" => "SUCCESS",
    		"data" => $data,
            "message" => $message
    	], 200);
    }

    public function failResponse($message) {
    	return response()->json([
    		"status" => "ERROR",
    		"message" => $message
    	], 500);
    }


    public function errorResponse($message, $errors = null, $errorCode = "ValidationError") {
    	return response()->json([
    		"status" => "FAIL",
    		"message" => $message,
    		"error" => [
    			"code" => $errorCode,
    			"message" => $message,
    			"errors" => $errors
    		]
    	], 500);
    }

    public function getPromoAvailability($code) {
		$promo = Promotion::whereStatus(1)
			->where('promotions.promo_code', $code)
			->where('promotions.date_start', '<=', now())
			->where('promotions.date_end','>=', now())->first();

        if(!$promo) {
            return [
                "availability" => false
            ];
        }

		$promotion_logs = PromotionLog::where('promo_id', $promo->id)->get();

		if(count($promotion_logs) > 0) {
			$quantity = (int)$promo->quantity - count($promotion_logs);
			$sum_points = 0;
			foreach($promotion_logs as $key => $promotion_log) {
				$sum_points += $promotion_log->points_amount;
			}
			$remaining_allocation = (double)$promo->allocation_amount - (double)$sum_points;

			if($quantity <= 0 || $remaining_allocation <= 0) {
				return [
					"availability" => false
				];
			}
		}

		$response = [
            "id" => $promo->id, 
			"promoCode" => $promo->promo_code, 
		    "availability" => true,
		    "points_as_payment" => $promo->points_as_payment,
		    "points_required" => $promo->points_required,
		    "remainingAllocation" => $promo->allocation_amount,
		    "dateExpiry" => $promo->date_end,
		];

		return $response;
	}

    // Store All Multiple Point Images
    public function storePointImage($request, $point_id) 
    {   
        if (isset($request->images)) {
            for ($i=0; $i < count($request->images); $i++) { 
                $image = "data:image/jpeg;base64, ".$request->images[$i]["image"];  // your base64 encoded
                $data = explode(',', $image);
                $current_timestamp = Carbon::now()->timestamp;
                $imageName = rand().'.jpg';
                $filenametostore = 'web/img/'. $imageName;

                $params = [
                    'point_id' => $point_id,
                    'title' => 'point_receipt',
                    'size' => strlen(base64_decode($request->images[$i]["image"])),
                    'path' => $filenametostore,
                ];

                PointImage::create($params);
                Storage::disk('s3')->put($filenametostore, base64_decode($data[1]));
            }
        }
    }

    public function getTimes($schedules) {
        $default_time = ["Closed", "Closed"];
        $schedule_monday = (isset($schedules["monday"])) ? str_replace(["[", "]"],"",explode("] - [", $schedules["monday"])) : $default_time;
        $schedule_tuesday = (isset($schedules["tuesday"])) ? str_replace(["[", "]"],"",explode("] - [", $schedules["tuesday"])) : $default_time;
        $schedule_wednesday = (isset($schedules["wednesday"])) ? str_replace(["[", "]"],"",explode("] - [", $schedules["wednesday"])) : $default_time;
        $schedule_thursday = (isset($schedules["thursday"])) ? str_replace(["[", "]"],"",explode("] - [", $schedules["thursday"])) : $default_time;
        $schedule_friday = (isset($schedules["friday"])) ? str_replace(["[", "]"],"",explode("] - [", $schedules["friday"])) : $default_time;
        $schedule_saturday = (isset($schedules["saturday"])) ? str_replace(["[", "]"],"",explode("] - [", $schedules["saturday"])) : $default_time;
        $schedule_sunday = (isset($schedules["sunday"])) ? str_replace(["[", "]"],"",explode("] - [", $schedules["sunday"])) : $default_time;

        return [
            "monday" => $schedule_monday,
            "tuesday" => $schedule_tuesday,
            "wednesday" => $schedule_wednesday,
            "thursday" => $schedule_thursday,
            "friday" => $schedule_friday,
            "saturday" => $schedule_saturday,
            "sunday" => $schedule_sunday
        ];
    }

    public function calculateRatings($reviews, $doctor_id) {
        $five_star_count = isset($reviews[$doctor_id]["5"]) ? count($reviews[$doctor_id]["5"]) : 0;
        $four_star_count = isset($reviews[$doctor_id]["4"]) ? count($reviews[$doctor_id]["4"]) : 0;
        $three_star_count = isset($reviews[$doctor_id]["3"]) ? count($reviews[$doctor_id]["3"]) : 0;
        $two_star_count = isset($reviews[$doctor_id]["2"]) ? count($reviews[$doctor_id]["2"]) : 0;
        $one_star_count = isset($reviews[$doctor_id]["1"]) ? count($reviews[$doctor_id]["1"]) : 0;
        // (5*252 + 4*124 + 3*40 + 2*29 + 1*33) / (252+124+40+29+33)
        $totalRatings = (5 * $five_star_count + 4*$four_star_count + 3*$three_star_count + 2*$two_star_count + 1*$one_star_count) / ($five_star_count+$four_star_count+$three_star_count+$two_star_count+$one_star_count);

        return $totalRatings;
    }

    public function countDecimals($number) {
    	return strlen(strrchr($number, '.'))-1;
    }

    public function addLogs($params) 
    {
        Log::create($params);
    }
}
