<?php

namespace App\Http\Controllers\Api\Hospital;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Establishment;
use App\Models\Speciality;
use App\Models\Doctor;

class ViewController extends Controller
{
    public function index() {
    	$establishments = Establishment::select('establishments.*')->join('merchant_types', 'establishments.merchant_type_id','merchant_types.id')->where('merchant_types.name', 'Medical')->get();

    	$data = [];
    	foreach ($establishments as $key => $establishment) {
			$data[] = [
				"id" => (string)$establishment->id,
		        "name" => $establishment->name,
		        "imageUrl" => isset($establishment->image->path) ? config('app.AWS_BUCKET_URL').$establishment->image->path : null,
		        "location" => [
		            "latitude" => (double)$establishment->latitude,
		            "longitude" => (double)$establishment->longitude,
		            "address" => $establishment->address
		        ]
			];
    	}

    	return $this->successResponse("success", $data);
    }

    public function getHospital($id) {
    	$establishment = Establishment::select('establishments.*')->join('merchant_types', 'establishments.merchant_type_id','merchant_types.id')->where('merchant_types.name', 'Medical')->where('establishments.id', $id)->first();
    	$specialties = [];
    	$doctors = [];
    	foreach ($establishment->doctors as $key => $doctor) {
    		$doctor_specialties = [];
    		foreach ($doctor->doctorSpecialities as $key => $specialty) {
    			$doctor_specialties[] = $specialty->specialty["name"];
    			$specialties[] = [
    				"id" => (string)$specialty->specialty["id"],
                	"name" => $specialty->specialty["name"]
    			];
    		}

    		$reviewsCount = 0;
    		$reviews = [];
    		$totalRatings = 0;
    		foreach ($doctor->doctorAppointments as $key => $doctor_appointments) {
    			if($doctor_appointments != null) {
                    if(isset($doctor_appointments->doctorReview->rate)) {
                        $reviewsCount++;
                        $reviews[$doctor->user_id][$doctor_appointments->doctorReview->rate][] = $doctor_appointments->doctorReview->rate;
                    }
    			}
    		}

    		if(count($reviews) > 0) {
    			$totalRatings = $this->calculateRatings($reviews, $doctor->user_id);
    		}

    		$doctors[] = [
    			"id" => (string)$doctor->user_id,
                "lastName" => $doctor->account->lastname,
                "firstName" => $doctor->account->firstname,
                "mi" => (isset($doctor->account->middlename[0])) ? $doctor->account->middlename[0] : null,
                "imageUrl" => isset($doctor->image->path) ? config('app.AWS_BUCKET_URL').$doctor->image->path : null,
                "specialities" => implode(",", $doctor_specialties),
                "hospitals" => [
                    [
                        "id" => (string)$establishment->id,
	        			"name" => $establishment->name,
                    ]
                ],
                "totalRatings" => $totalRatings,
                "consultationFee" => $doctor->minimum_consultation_fee.' - '.$doctor->maximum_consultation_fee,
                "reviewsCount" => $reviewsCount
    		];
    	}

    	$specialty_temp = array_unique(array_column($specialties, 'name'));
    	$specialties = array_intersect_key($specialties, $specialty_temp);

    	$data = [
    		"id" => (string)$establishment->id,
	        "name" => $establishment->name,
	        "imageUrl" => isset($establishment->image->path) ? config('app.AWS_BUCKET_URL').$establishment->image->path : null,
	        "location" => [
	            "latitude" => (double)$establishment->latitude,
	            "longitude" => (double)$establishment->longitude,
	            "address" => $establishment->address
	        ],
	        "description" => $establishment->description,
	        "specialities" => $specialties,
	        "doctors" => $doctors,
            "timings" => [
				"times" => array_values($this->getTimes($establishment->establishment_schedule)),
				"currentDayTimeIndex" => (int) date('w', strtotime('Today'))
			]
    	];

    	return $this->successResponse("success",$data);
    }

    public function search(Request $request) {
    	$specialities = Speciality::select('specialities.*')
    				->join('speciality_types','specialities.speciality_type_id', 'speciality_types.id')
    				->where('speciality_types.name','Medical')
    				->where('specialities.status',1)
    				->where('specialities.name','like', '%'.$request->q.'%')->get();

    	$specialities_data = [];
    	foreach ($specialities as $key => $speciality) {
    		$specialities_data[] = [
    			"id" => (string)$speciality->id,
    			"name" => $speciality->name,
    			"imageUrl" => isset($speciality->image->path) ? config('app.AWS_BUCKET_URL').$speciality->image->path : null,
    		];
    	}

    	$hospitals_data = [];
    	$hospitals = Establishment::select('establishments.*')->join('merchant_types', 'establishments.merchant_type_id','merchant_types.id')->where('merchant_types.name', 'Medical')->where('establishments.name', 'like', '%'.$request->q.'%')->get();

    	foreach ($hospitals as $key => $hospital) {
    		$hospitals_data[] = [
    			"id" => (string)$hospital->id,
		        "name" => $hospital->name,
		        "imageUrl" => isset($hospital->image->path) ? config('app.AWS_BUCKET_URL').$hospital->image->path : null,
		        "location" => [
		            "latitude" => (double)$hospital->latitude,
		            "longitude" => (double)$hospital->longitude,
		            "address" => $hospital->address
		        ]
    		];
    	}

    	$doctors_data = [];
    	$doctors = Doctor::whereHas('account', function($query) use ($request) {
                        $query->where('firstname', 'LIKE', '%'.$request->q.'%')
                            ->orWhere('lastname',  'LIKE', '%'.$request->q.'%')
                            ->orWhere('middlename',  'LIKE', '%'.$request->q.'%');
                    })->get();

    	foreach ($doctors as $key => $doctor) {
    		$doctor_specialties = [];
    		foreach ($doctor->doctorSpecialities as $key => $specialty) {
    			$doctor_specialties[] = $specialty->specialty["name"];
    		}

    		$establishments = [];
    		foreach ($doctor->merchant->establishments as $key => $establishment) {
    			$establishments[] = [
    				"id" => (string)$establishment->id,
                    "name" => $establishment->name
    			];
    		}

    		$reviews = [];
    		$totalRatings = 0;
    		foreach ($doctor->doctorAppointments as $key => $doctor_appointments) {
    			if($doctor_appointments != null) {
                    if(isset($doctor_appointments->doctorReview->rate)) {
    				    $reviews[$doctor->user_id][$doctor_appointments->doctorReview->rate][] = $doctor_appointments->doctorReview->rate;
                    }
    			}
    		}

    		if(count($reviews) > 0) {
    			$totalRatings = $this->calculateRatings($reviews, $doctor->user_id);
    		}

    		$accredited_hmos = [];
    		foreach ($doctor->doctorPaymentMethods as $key => $payment_method) {
    			$accredited_hmos[] = [
    				"id" => (string)$payment_method->id,
    				"name" => ucwords(str_replace("_"," ",$payment_method->name))
    			];
    		}

    		$doctors_data[] = [
    			"id" => (string)$doctor->user_id,
                "firstName" => $doctor->account->firstname,
                "mi" => (isset($doctor->account->middlename[0])) ? $doctor->account->middlename[0] : null,
                "lastName" => $doctor->account->lastname,
                "imageUrl" => isset($doctor->image->path) ? config('app.AWS_BUCKET_URL').$doctor->image->path : null,
                "specialities" => implode(",", $doctor_specialties),
                "hospitals" => $establishments,
                "totalRatings" => $totalRatings,
                "consultationFee" => $doctor->minimum_consultation_fee.' - '.$doctor->maximum_consultation_fee,
                "accreditedHMOs" => $accredited_hmos
    		];
    	}

    	$data = [
    		"hospitals" => $hospitals_data,
    		"specialities" => $specialities_data,
    		"doctors" => $doctors_data
    	];
    	return $this->successResponse("success",$data);
    }
}
